package ru.yandex.search.district.indexer;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.search.district.DistrictChangeType;
import ru.yandex.search.district.DistrictConstants;
import ru.yandex.search.district.DistrictFields;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class DistrictIndexUpdateHandler
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final String DISTRICTS = "districts";

    private final DistrictIndexerProxy proxy;
    private final Map<DistrictChangeType, DistrictIndexHandler> handlers;
    private final TimeFrameQueue<DistrictChangeType> changeTypes;

    public DistrictIndexUpdateHandler(final DistrictIndexerProxy proxy) {
        this.proxy = proxy;

        this.handlers = new LinkedHashMap<>();
        this.handlers.put(DistrictChangeType.ADD, new AddDistrictHandler());
        this.handlers.put(
            DistrictChangeType.UPDATE,
            new UpdateDistrictHandler());
        this.handlers.put(
            DistrictChangeType.DELETE,
            new DeleteDistrictHandler());
        this.handlers.put(
            DistrictChangeType.DISLIKE,
            new IncrementDistrictHandler(DistrictFields.DISLIKES_COUNT));
        this.handlers.put(
            DistrictChangeType.LIKE,
            new IncrementDistrictHandler(DistrictFields.LIKES_COUNT));
        this.handlers.put(
            DistrictChangeType.VIEW,
            new IncrementDistrictHandler(DistrictFields.VIEWS_CNT));

        this.changeTypes =
            new TimeFrameQueue<>(proxy.config().metricsTimeFrame());

        this.proxy.registerStater(
            new PassiveStaterAdapter<>(
                changeTypes,
                new EnumStaterFactory<>(
                    type -> "change-type-" + type + "_ammm",
                    DistrictChangeType.values())));
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            return new JsonAsyncTypesafeDomConsumer(
                ((HttpEntityEnclosingRequest) request).getEntity(),
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            throw new BadRequestException("Payload expected");
        }
    }

    private DistrictChangeType handleMultiDistrict(
        final ProxySession session,
        final List<JsonObject> districts,
        final JsonMap item,
        final Long cityId,
        final BasicDistrictIndexer indexer,
        final BasicDistrictIndexer cityIndexer)
        throws HttpException, IOException, JsonException
    {
        DistrictChangeType changeType = null;
        for (JsonObject id: districts) {
            long prefix = id.asLong();
            if (prefix < 0) {
                session.logger().log(
                    Level.WARNING,
                    "One of districts has "
                        + "negative shard number "
                        + JsonType.HUMAN_READABLE.toString(
                        item));
                prefix = 0;
            }
            // first we do it for district base index
            AbstractIndexContext indexContext =
                new DistrictBaseIndexContext(
                    prefix,
                    session,
                    item);

            handlers.get(indexContext.changeType()).handle(
                indexContext,
                indexer);

            if (cityId == null) {
                session.logger().warning(
                    "City_Id is null "
                        + JsonType.NORMAL.toString(
                        item));
            } else {
                //than for city base index
                indexContext =
                    new DistrictCityBaseContext(
                        id.asLong(),
                        session,
                        item);
                handlers.get(indexContext.changeType()).handle(
                    indexContext,
                    cityIndexer);
            }

            changeType = indexContext.changeType();
        }

        return changeType;
    }

    private DistrictChangeType handleSingleDistrict(
        final ProxySession session,
        final JsonObject districtId,
        final JsonMap item,
        final Long cityId,
        final BasicDistrictIndexer indexer,
        final BasicDistrictIndexer cityIndexer)
        throws HttpException, IOException, JsonException
    {
        long prefix = districtId.asLong();
        if (prefix < 0) {
            prefix = 0;
        }
        AbstractIndexContext indexContext =
            new DistrictBaseIndexContext(
                prefix,
                session,
                item);

        if (indexContext.shard() < 0) {
            session.logger().log(
                Level.WARNING,
                "District id is negative for  "
                    + JsonType.HUMAN_READABLE.toString(item));
        }

        handlers.get(indexContext.changeType()).handle(
            indexContext,
            indexer);
        if (cityId == null) {
            session.logger().warning(
                "CityId is null " + JsonType.NORMAL.toString(
                    item));
        } else {
            //than for city base index
            indexContext =
                new DistrictCityBaseContext(
                    districtId.asLong(),
                    session,
                    item);
            handlers.get(indexContext.changeType()).handle(
                indexContext,
                cityIndexer);
        }

        return indexContext.changeType();
    }

    // https://st.yandex-team.ru/PS-3657
    private DistrictChangeType handleNoDistrict(
        final ProxySession session,
        final JsonMap item,
        final BasicDistrictIndexer cityIndexer)
        throws HttpException, IOException, JsonException
    {
        DistrictCityBaseContext indexContext =
            new DistrictCityBaseContext(
                -1L,
                session,
                item);
        handlers.get(indexContext.changeType()).handle(
            indexContext,
            cityIndexer);
        return indexContext.changeType();
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(proxy, exchange, context);

        DoubleFutureCallback<Object, Object> dcb =
            new DoubleFutureCallback<>(new IndexCallback(session));
        BasicDistrictIndexer indexer = new BasicDistrictIndexer(
            proxy,
            session,
            DistrictConstants.DISTRICT_QUEUE,
            dcb.first());
        BasicDistrictIndexer cityIndexer = new BasicDistrictIndexer(
            proxy,
            session,
            DistrictConstants.DISTRICT_CITY_QUEUE,
            dcb.second());

        long[] stat = new long[DistrictChangeType.values().length];
        try {
            JsonList list = payload.asList();
            session.logger().info("Updating docs size " + list.size());

            for (int i = 0; i < list.size(); i++) {
                JsonMap item = list.get(i).asMap();
                JsonMap fields = item.getMap("fields");
                try {
                    JsonList districts = fields.getListOrNull(DISTRICTS);
                    Long cityId =
                        fields.getLong("city_id", null);
                    JsonObject districtId =
                        fields.getOrDefault(
                            DistrictFields.DISTRICT_ID.field(),
                            null);

                    DistrictChangeType changeType = null;

                    if ((districts == null || districts.size() == 0)
                        && (districtId == null || districtId == JsonNull.INSTANCE))
                    {
                        if (cityId == null) {
                            throw new BadRequestException(
                                "No city no districts, lonely, lost child, where should i place you?");
                        }

                        changeType = handleNoDistrict(session, item, cityIndexer);
                    } else {
                        if (districts != null && districts.size() != 0) {
                            changeType = handleMultiDistrict(session, districts, item, cityId, indexer, cityIndexer);
                        } else {
                            changeType = handleSingleDistrict(session, districtId, item, cityId, indexer, cityIndexer);
                        }
                    }

                    if (changeType != null) {
                        changeTypes.accept(changeType);
                        stat[changeType.ordinal()] += 1;
                    }

                    if (proxy.indexerConfig().printUpdates()
                        && (changeType == DistrictChangeType.ADD
                        || changeType == DistrictChangeType.UPDATE))
                    {
                        session.logger().info("Update " + JsonType.NORMAL.toString(item));
                    }
                } catch (JsonException je) {
                    throw new BadRequestException(
                        "Failed to handle doc "
                            + JsonType.HUMAN_READABLE.toString(item),
                        je);
                }
            }
            indexer.done();
            cityIndexer.done();
            StringBuilder statSb = new StringBuilder("ChangeType stat ");
            for (int i = 0; i < stat.length; i++) {
                if (stat[i] > 0) {
                    statSb.append(DistrictChangeType.values()[i]);
                    statSb.append(": ");
                    statSb.append(stat[i]);
                    statSb.append(";");
                }
            }
            session.logger().info(statSb.toString());
        } catch (JsonException | IOException e) {
            throw new BadRequestException("Bad input data/format", e);
        }
    }

    private static final class IndexCallback
        extends AbstractProxySessionCallback<Object>
    {
        private IndexCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Object o) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
