package ru.yandex.search.district;

import java.io.IOException;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.search.district.config.ImmutableDistrictSearchProxyConfig;
import ru.yandex.search.district.search.DistrictSearchHandler;
import ru.yandex.search.district.suggest.DistrictSuggestHandler;
import ru.yandex.search.district.wizard.DistrictWizardSearchHandler;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class DistrictSearchProxy
    extends UniversalSearchProxy<ImmutableDistrictSearchProxyConfig>
{
    private final AsyncClient wizardSerchClient;
    private final SharedConnectingIOReactor wizardReactor;
    private final TimeFrameQueue<Integer> wizardEmptyResponses;
    private final TimeFrameQueue<Integer> searchEmptyResponses;

    public DistrictSearchProxy(
        final ImmutableDistrictSearchProxyConfig config)
        throws IOException
    {
        super(config);

        wizardEmptyResponses = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                wizardEmptyResponses,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "wizard-empty-responses_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "wizard-responses_ammm",
                        CountAggregatorFactory.INSTANCE))));

        searchEmptyResponses = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                searchEmptyResponses,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "search-empty-responses_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "search-responses_ammm",
                        CountAggregatorFactory.INSTANCE))));

        wizardReactor = new SharedConnectingIOReactor(
            config,
            config.dnsConfig(),
            new ThreadGroup(getThreadGroup(), "Wizzard-Client"));
        closeChain.add(wizardReactor);

        wizardSerchClient =
            registerClient(
                "WizzardClient",
                new AsyncClient(
                    wizardReactor,
                    config.wizardSearchClientConfig()),
                config.wizardSearchClientConfig());

        register(
            new Pattern<>("/api/district/search", true),
            new DistrictSearchHandler(this));

        register(
            new Pattern<>("/api/district/suggest/popular", true),
            new DistrictSuggestHandler(this));

        register(
            new Pattern<>("/api/district/wizard/search", true),
            new DistrictWizardSearchHandler(this));
    }

    @Override
    public void start() throws IOException {
        super.start();

        wizardReactor.start();
        wizardSerchClient.start();
    }

    public AsyncClient wizardSerchClient() {
        return wizardSerchClient;
    }

    public void wizardResponse(final int count) {
        if (count > 0) {
            wizardEmptyResponses.accept(0);
        } else {
            wizardEmptyResponses.accept(1);
        }
    }

    public void searchResponse(final int count) {
        if (count > 0) {
            searchEmptyResponses.accept(0);
        } else {
            searchEmptyResponses.accept(1);
        }
    }
}
