package ru.yandex.search.district.search;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.client.protocol.HttpClientContext;
import org.joda.time.DateTimeZone;

import ru.yandex.collection.IntInterval;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.LongParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.district.DistrictConstants;
import ru.yandex.search.district.DistrictEntityType;
import ru.yandex.search.district.DistrictSearchProxy;
import ru.yandex.search.district.Rank;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class BasicDistrictSearchContext
    implements DistrictSearchContext,
    RequestProvider,
    DistrictSearchContextProvider
{
    private static final int DEFAULT_LENGTH = 30;
    private static final int TZ_MLTPL = 60000;

    private static final DateTimeZone MSK_TIMEZONE =
        DateTimeZone.forID("Europe/Moscow");

    private static final User DEFAULT_USER =
        new User(DistrictConstants.DISTRICT_QUEUE, new LongPrefix(1L));
    private static final CollectionParser<Long, Set<Long>, Exception>
        DS_PARSER =
        new CollectionParser<>(
            LongParser.INSTANCE,
            LinkedHashSet::new);
    private static final CollectionParser<Long, Set<Long>, Exception>
        LONG_SET_PARSER =
        new CollectionParser<>(
            LongParser.INSTANCE,
            LinkedHashSet::new);

    private static final CollectionParser<String, Set<String>, Exception>
        STR_SET_PARSER =
        new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            LinkedHashSet::new);

    private final ProxySession session;
    private final JsonType jsonType;
    private final AsyncClient client;
    private final Supplier<? extends HttpClientContext> contextGenerator;
    private final String request;
    private final DistrictSearchProxy proxy;
    private final DistrictEntityType entityType;
    private final Collection<Long> districts;
    private final Collection<Long> tags;
    private final int length;
    private final int lengthWithOffset;
    private final int offset;
    private final boolean allowLaggingHosts;
    private final boolean saveRequest;
    private final Rank rank;
    private final Set<Long> cityIds;
    private final Set<Long> mentionUsers;
    private final SearchScope searchScope;
    private final String user;
    private final Collection<String> get;
    private final DistrictSearchSession searchSession;
    private final DateTimeZone timezone;
    private final IntInterval timeRange;
    private final User indexUser;

    // should be
    private final boolean returnSession;

    public BasicDistrictSearchContext(
        final DistrictSearchProxy proxy,
        final ProxySession session)
        throws BadRequestException, IOException
    {
        this.proxy = proxy;
        this.session = session;

        CgiParams params = session.params();
        Collection<Long> districts =
            params.get(
                "districts",
                Collections.emptySet(),
                DS_PARSER);
        if (districts.isEmpty()) {
            districts =
                params.getAll("district", Collections.emptySet(), DS_PARSER);
        }

        get = params.get("get", null, STR_SET_PARSER);

        tags = params.getAll("tags", Collections.emptySet(), DS_PARSER);

        this.cityIds = params.get("city", Collections.emptySet(), LONG_SET_PARSER);
        this.mentionUsers = params.get("mention_users", Collections.emptySet(), LONG_SET_PARSER);

        if (!cityIds.isEmpty()) {
            searchScope = SearchScope.CITY;
            this.indexUser = new User(
                DistrictConstants.DISTRICT_CITY_QUEUE,
                new LongPrefix(cityIds.iterator().next()));
        } else {
            if (districts.size() > 1) {
                throw new BadRequestException(
                    "No city id and more than one district");
            }
            searchScope = SearchScope.DISTRICTS;
            this.indexUser = DEFAULT_USER;
        }

        this.districts = districts;
        this.rank = params.getEnum(Rank.class, "sort", Rank.RELEVANCE);

        offset = params.get(
            "offset",
            0,
            NonNegativeIntegerValidator.INSTANCE);

        String sessionId = params.getString("session", null);
        if (sessionId == null) {
            sessionId =
                session.headers().getLastOrNull(
                    "X-Yandex-District-Search-Session");
        }

        this.searchSession = new DistrictSearchSession(sessionId, offset);
        if (sessionId != null) {
            logger().info(
                "Previous we found docs cnt "
                    + this.searchSession.getDocs().size());
        }

        timezone = params.get(
            "tzoffset",
            MSK_TIMEZONE,
            x -> DateTimeZone.forOffsetMillis(Integer.parseInt(x) * TZ_MLTPL));

        int from = params.get(
            "time_from",
            0,
            NonNegativeIntegerValidator.INSTANCE);
        int to = params.get(
            "time_to",
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);

        if ((from > 0 || to != Integer.MAX_VALUE) && to >= from) {
            this.timeRange = new IntInterval(from, to);
        } else {
            this.timeRange = null;
        }

        length = params.get(
            "length",
            DEFAULT_LENGTH,
            PositiveIntegerValidator.INSTANCE);

        lengthWithOffset = offset + length;

        entityType =
            params.getEnum(
                DistrictEntityType.class,
                "scope",
                DistrictEntityType.ALL);

        user = params.getString("user", null);

        // should we return session in response
        returnSession = params.getBoolean("with-session", true);

        allowLaggingHosts =
            params.getBoolean("allow-lagging-hosts", true);

        request = params.getString("request");

        client = proxy.searchClient().adjust(session.context());
        contextGenerator =
            session.listener().createContextGeneratorFor(client);
        jsonType = JsonTypeExtractor.NORMAL.extract(params);
        saveRequest = params.getBoolean("save-request", false);
    }

    public DateTimeZone timezone() {
        return timezone;
    }

    public Collection<String> get() {
        return get;
    }

    @Override
    public User user() {
        return indexUser;
    }

    public String districtUser() {
        return user;
    }

    @Override
    public Long minPos() {
        return null;
    }

    @Override
    public AbstractAsyncClient<?> client() {
        return client;
    }

    @Override
    public Logger logger() {
        return session.logger();
    }

    @Override
    public long lagTolerance() {
        return allowLaggingHosts ? Long.MAX_VALUE : 0L;
    }

    public ProxySession session() {
        return session;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public Supplier<? extends HttpClientContext> contextGenerator() {
        return contextGenerator;
    }

    @Override
    public String request() {
        return request;
    }

    public DistrictSearchProxy proxy() {
        return proxy;
    }

    public DistrictEntityType scope() {
        return entityType;
    }

    public Collection<Long> districts() {
        return districts;
    }

    public int length() {
        return lengthWithOffset;
    }

    public int outerLength() {
        return length;
    }

    public int offset() {
        return offset;
    }

    @Override
    public BasicDistrictSearchContext searchContext() {
        return this;
    }

    public Rank rank() {
        return rank;
    }

    public Set<Long> cityIds() {
        return cityIds;
    }

    public Set<Long> mentionUsers() {
        return mentionUsers;
    }

    public SearchScope searchScope() {
        return searchScope;
    }

    public Collection<Long> tags() {
        return tags;
    }

    public boolean saveRequest() {
        return saveRequest;
    }

    public DistrictSearchSession searchSession() {
        return searchSession;
    }

    public boolean returnSession() {
        return returnSession;
    }

    public boolean hasTimeRange() {
        return timeRange != null;
    }

    public IntInterval timeRange() {
        return timeRange;
    }
}
