package ru.yandex.search.district.search;

import java.io.IOException;
import java.util.List;

import ru.yandex.collection.IntInterval;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.LongParser;
import ru.yandex.search.district.DistrictFields;
import ru.yandex.util.string.StringUtils;

public class DistrictResultItem implements JsonValue {
    private final JsonMap data;
    private final long likesCnt;
    private final long dislikesCnt;
    private final long viewsCnt;
    private final long createdAt;
    private final long commentsCnt;
    private final long fetchTime;
    private final double rank;
    private final long hitRank;
    private final List<IntInterval> highlights;
    private final String id;
    private final String itemId;
    private final Long entityId;

    // CSOFF: ParameterNumber
    public DistrictResultItem(
        final JsonMap data,
        final List<IntInterval> highlights,
        final List<String> hitFields,
        final long fetchTime)
        throws JsonException
    {
        this.data = data;
        this.fetchTime = fetchTime;
        this.highlights = highlights;
        this.id = data.getString(DistrictFields.ID.field());
        this.viewsCnt = data.getLong(DistrictFields.VIEWS_CNT.field(), 0L);
        this.likesCnt =
            data.getLong(DistrictFields.LIKES_COUNT.field(), 0L);
        this.dislikesCnt =
            data.getLong(DistrictFields.DISLIKES_COUNT.field(), 0L);
        this.createdAt = data.getLong(DistrictFields.CREATE_DATE.field());
        this.commentsCnt =
            data.getLong(DistrictFields.COMMENTS_COUNT.field(), 0L);
        this.rank = calcRank();
        this.entityId =
            data.get(DistrictFields.ENTITY_ID.field(), -1L, LongParser.INSTANCE);
        this.data.replace(DistrictFields.ENTITY_ID.field(), new JsonLong(entityId));
        // https://st.yandex-team.ru/PS-3657
        long districtId = this.data.getLong(DistrictFields.DISTRICT_ID.field(), -1L);
        if (districtId < 0) {
            this.data.remove(DistrictFields.DISTRICT_ID.field());
        }

        this.itemId =
            StringUtils.concat(data.getString(DistrictFields.ENTITY_TYPE
                    .field()),
                '_',
                String.valueOf(entityId));
        JsonList hl = new JsonList(BasicContainerFactory.INSTANCE);
        for (IntInterval interval: highlights) {
            JsonList intervalJson
                = new JsonList(BasicContainerFactory.INSTANCE);
            intervalJson.add(new JsonLong(interval.min()));
            intervalJson.add(new JsonLong(interval.max()));
            hl.add(intervalJson);
        }

        this.data.put("text_highlighted", hl);

        long hitRank = 0;
        for (int i = 0; i < hitFields.size(); i++) {
            int value = data.getInt(hitFields.get(i), 0);
            if (value > 0) {
                hitRank += value << (hitFields.size() - i);
            }
        }

        this.hitRank = hitRank;
    }
    // CSON: ParameterNumber

    public JsonMap data() {
        return data;
    }

    public long likesCnt() {
        return likesCnt;
    }

    public long dislikesCnt() {
        return dislikesCnt;
    }

    public long viewsCnt() {
        return viewsCnt;
    }

    public long createdAt() {
        return createdAt;
    }

    public long commentsCnt() {
        return commentsCnt;
    }

    public long fetchTime() {
        return fetchTime;
    }

    public double rank() {
        return rank;
    }

    public long fuzzyHitRank() {
        return hitRank;
    }

    public List<IntInterval> highlights() {
        return highlights;
    }

    public Long entityId() {
        return entityId;
    }

    public String id() {
        return id;
    }

    //CSOFF: MagicNumber
    private double calcRank() {
        double ageMinutes = (fetchTime - createdAt) / 60 + 1;

        return 100.0 * commentsCnt / (viewsCnt + 1)
            + 50.0 * likesCnt / (viewsCnt + 1)
            - Math.log(1 + ageMinutes)
            - 60.0 * dislikesCnt / (viewsCnt + 1);
    }
    //CSON: MagicNumber

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        data.writeValue(writer);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof DistrictResultItem)) {
            return false;
        }

        DistrictResultItem item = (DistrictResultItem) o;

        return itemId.equals(item.itemId);
    }

    @Override
    public int hashCode() {
        return itemId.hashCode();
    }

    public String itemId() {
        return itemId;
    }
}
