package ru.yandex.search.district.search;

import java.io.IOException;
import java.util.ArrayList;
import java.util.logging.Level;

import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.district.Rank;

public class DistrictSearchResultPrinter
    extends AbstractProxySessionCallback<DistrictSearchResult>
{
    private final BasicDistrictSearchContext context;

    public DistrictSearchResultPrinter(
        final BasicDistrictSearchContext context)
    {
        super(context.session());

        this.context = context;
    }

    @Override
    public void completed(final DistrictSearchResult result) {
        session.connection().setSessionInfo(
            SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
            Long.toString(result.items().size()));

        StringBuilderWriter sbw = new StringBuilderWriter();

        ArrayList<DistrictResultItem> results = new ArrayList<>(result.items());
        if (context.rank() == Rank.RELEVANCE) {
            results.sort(RankComparator.INSTANCE);
        } else {
            results.sort(CreateTimeComparator.INSTANCE);
        }
        int max =
            Math.min(context.length(), results.size());

        try (JsonWriter writer = context.jsonType().create(sbw)) {
            writer.startObject();
            writer.key("total");
            if (results.size() < context.length()) {
                writer.value(results.size());
            } else {
                writer.value(result.totalResults());
            }

            writer.key("results");
            writer.startArray();
            context.logger().info(
                "Printing from " + context.offset() + " to "
                    + max + " results " + results.size());
            DistrictSearchSession searchSession = context.searchSession();
            if (context.get() == null) {
                int count = 0;
                int i = 0;
                while (count < context.outerLength() && i < results.size()) {
                    DistrictResultItem item = results.get(i);
                    if (searchSession.add(item)) {
                        item.writeValue(writer);
                        count += 1;
                    } else {
                        context.logger().info("Filtering " + item.id());
                    }

                    i++;
                }
            } else {
                int count = 0;
                int i = 0;
                while (count < context.outerLength() && i < results.size()) {
                    DistrictResultItem item = results.get(i);
                    if (searchSession.add(item)) {
                        context.logger().info("Adding " + item.id());
                        count += 1;

                        JsonMap map = item.data();
                        writer.startObject();
                        for (String field: context.get()) {
                            JsonObject obj = map.get(field);
                            if (obj != null) {
                                writer.key(field);
                                writer.value(obj);
                            }
                        }

                        writer.endObject();
                    } else {
                        context.logger().info(
                            "Filtering out " + item.id());
                    }

                    i++;
                }
            }

            writer.endArray();
            if (context.returnSession()) {
                writer.key("session");
                writer.value(searchSession.getSessionId());
            }

            writer.endObject();
        } catch (IOException ioe) {
            failed(ioe);
            return;
        }

        session.response(
            HttpStatus.SC_OK,
            new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(context.session().acceptedCharset())));
        context.proxy().searchResponse(results.size());

        if (context.saveRequest()) {
            ProducerClient client =
                context.proxy().producerClient().adjust(session.context());
            try {
                QueryConstructor qc =
                    new QueryConstructor(
                        context.proxy().producerHost()
                            + "/api/district/index/request?");
                qc.append("request", context.request());

                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(qc.toString());
                generator.copyHeader(
                    session.request(),
                    YandexHeaders.X_YA_SERVICE_TICKET);
                client.execute(
                    context.proxy().producerHost(),
                    generator,
                    new Callback());
            } catch (BadRequestException bre) {
                session.logger().log(
                    Level.WARNING,
                    "Failed to create save-request",
                    bre);
            }
        } else {
            context.logger().info("Saving request disabled");
        }
    }

    private final class Callback implements FutureCallback<Object> {
        @Override
        public void completed(final Object o) {
            context.logger().info(context.request() + " saved");
        }

        @Override
        public void failed(final Exception e) {
            context.logger().warning(context.request() + " saving failed");
            context.logger().log(Level.WARNING, "Request save Failed", e);
        }

        @Override
        public void cancelled() {
        }
    }
}
