package ru.yandex.search.district.search;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.IntInterval;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.district.AbstractDistrictSearchRule;
import ru.yandex.search.district.DistrictConstants;
import ru.yandex.search.district.DistrictEntityType;
import ru.yandex.search.district.DistrictFields;
import ru.yandex.search.district.DistrictSearchProxy;
import ru.yandex.search.district.Rank;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.ps.highlight.IntervalHighlighter;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class DistrictSearchRule
    <T extends RequestProvider & DistrictSearchContextProvider>
    extends AbstractDistrictSearchRule
    implements SearchRule<T, DistrictSearchResult>
{
    private static final String LOCALE = "locale";
    private static final String PREFIX = "prefix";
    private static final String SERVICE = "service";
    private static final int SORT_BY_RANK_THRESHOLD = 3;
    private static final int PRUINNING_LENGTH_MULTIPLIER = 10;
    private static final int PRUINNING_LENGTH_THRESHOLD = 1000;
    private static final String DEFAULT_GET_FIELDS =
        DistrictFields.ENTITY_ID.field() + ','
            + DistrictFields.ENTITY_TYPE.field() + ','
            + DistrictFields.COMMENTS_COUNT.field() + ','
            + DistrictFields.CREATE_DATE.field() + ','
            + DistrictFields.DISLIKES_COUNT.field() + ','
            + DistrictFields.LIKES_COUNT.field() + ','
            + DistrictFields.VIEWS_CNT.field() + ','
            + DistrictFields.ID.field() + ','
            + DistrictFields.TEXT.field();
    private static final String TEXT = "text";

    private static final String EXACT_POSTFIX = "_exact_hits";
    private static final String NON_EXACT_POSTFIX = "_non_exact_hits";
    private static final String LENGTH = "length";
    private static final Map<SearchScope, Collection<String>> SEARCH_FIELDS;
    private static final Map<SearchScope, String> PERFIELD;
    private static final Map<SearchScope, List<String>> SORT_FIELDS;

    static {
        Map<SearchScope, Collection<String>> fields = new LinkedHashMap<>();
        Map<SearchScope, List<String>> sortFields = new LinkedHashMap<>();
        Map<SearchScope, String> perfield = new LinkedHashMap<>();
        fields.put(
            SearchScope.CITY,
            Arrays.asList(
                DistrictFields.TEXT_WS.prefixedField(),
                DistrictFields.TEXT.prefixedField()));
        fields.put(
            SearchScope.DISTRICTS,
            Arrays.asList(
                DistrictFields.TEXT_WS.prefixedField(),
                DistrictFields.TEXT.prefixedField()));
        SEARCH_FIELDS = Collections.unmodifiableMap(fields);

        for (Map.Entry<SearchScope, Collection<String>> entry
            : fields.entrySet())
        {
            perfield.put(
                entry.getKey(),
                "perfield(" + String.join(",", entry.getValue()) + ')');
            List<String> sortCol = new ArrayList<>();
            for (String k: entry.getValue()) {
                sortCol.add('#' + k + EXACT_POSTFIX);
                sortCol.add('#' + k + NON_EXACT_POSTFIX);
            }

            sortFields.put(
                entry.getKey(),
                Collections.unmodifiableList(sortCol));
        }

        PERFIELD = Collections.unmodifiableMap(perfield);
        SORT_FIELDS = Collections.unmodifiableMap(sortFields);
    }

    private final Timer timer = new Timer("SearchTimer", true);

    public DistrictSearchRule(
        final DistrictSearchProxy proxy)
    {
    }

    // CSOFF: ParameterNumber
    private void executeDistricts(
        final BasicDistrictSearchContext context,
        final AggregateCallback callback,
        final QueryConstructor query,
        final T input)
        throws BadRequestException
    {
        Rank rank = context.rank();
        SearchRequestText fuzzyText = SearchRequestText.parseSuggest(
            input.request(),
            context.session().params().getLocale(LOCALE, Locale.ROOT));

        SearchRequestText nonFuzzyText =
            new SearchRequestText(input.request());

        StringBuilder querySb = query.sb();
        int baseLength = querySb.length();
        MultiFutureCallback<JsonObject> mfcb =
            new MultiFutureCallback<>(callback);
        for (Long districId: context.districts()) {
            List<HttpHost> hosts =
                context.proxy().searchMap().searchHosts(
                    new User(
                        DistrictConstants.DISTRICT_QUEUE,
                        new LongPrefix(districId)));

            StringBuilder text = text(context);

            if (!fuzzyText.hasWords() && !nonFuzzyText.hasWords()) {
                if (text.length() > 0) {
                    text.append(AND);
                }
                text.append(DistrictFields.DISTRICT_ID.prefixedField());
                text.append(':');
                text.append(districId);

                rank = Rank.DATE;
            } else {
                if (text.length() > 0) {
                    text.append(AND);
                }
                text.append('(');
                text.append('(');
                if (fuzzyText.hasWords()) {
                    String fuzzyField = DistrictFields.TEXT.prefixedField();
                    if (fuzzyText.hasWords()) {
                        fuzzyText.fieldsQuery(text, fuzzyField);
                        fuzzyText.negationsQuery(text, fuzzyField);
                    }

                    if (!nonFuzzyText.isEmpty()) {
                        text.append(')');
                        text.append(OR);
                        text.append('(');
                    }

                    if (fuzzyText.text().length() <= SORT_BY_RANK_THRESHOLD) {
                        rank = Rank.DATE;
                    }
                }

                if (!nonFuzzyText.isEmpty()) {
                    String wsField = DistrictFields.TEXT_WS.prefixedField();
                    if (nonFuzzyText.hasWords()) {
                        nonFuzzyText.fieldsQuery(text, wsField);
                        nonFuzzyText.negationsQuery(text, wsField);
                    }

                    if (nonFuzzyText.text().length()
                        <= SORT_BY_RANK_THRESHOLD)
                    {
                        rank = Rank.DATE;
                    }
                }

                text.append(')');
                text.append(')');
            }

            query.append(TEXT, text.toString());
            query.append(SERVICE, context.user().service());
            query.append(PREFIX, context.districts().iterator().next());

            if (context.rank() == Rank.RELEVANCE && rank == Rank.DATE) {
                query.append(LENGTH, context.length());
                context.session().logger().info(
                    "Request to small, reducing search to date order "
                        + fuzzyText.text() + ' ' + nonFuzzyText.text());
            } else {
                query.append(
                    LENGTH,
                    Math.min(
                        PRUINNING_LENGTH_MULTIPLIER * context.length(),
                        PRUINNING_LENGTH_THRESHOLD));
            }

            applyRank(query, context, rank);

            String uri = query.toString();
            context.searchContext().logger().info(uri);
//            context.proxy().parallelRequest(
//                context.session(),
//                hosts,
//                context,
//                new BasicAsyncRequestProducerGenerator(uri),
//                JsonAsyncTypesafeDomConsumerFactory.OK,
//                context.contextGenerator(),
//                new ParallelRequestCallback<>(
//                    context,
//                    mfcb.newCallback(),
//                    hosts,
//                    -1L));
            context.proxy().sequentialRequest(
                context.session(),
                context,
                new BasicAsyncRequestProducerGenerator(uri),
                100L,
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.contextGenerator(),
                mfcb.newCallback()
            );

            querySb.setLength(baseLength);
        }

        mfcb.done();
    }

    private void executeCity(
        final BasicDistrictSearchContext context,
        final AggregateCallback callback,
        final QueryConstructor query,
        final T input)
        throws BadRequestException
    {
        if (context.cityIds().size() == 1) {
            executeSingleCity(context, context, callback, query, input);
        } else {
            MultiFutureCallback<List<JsonObject>> mfcb
                = new MultiFutureCallback<>(
                    new MultiCityMergeCallback(callback));
            for (Long cityId: context.cityIds()) {
                executeSingleCity(
                    context,
                    new SingleCityContext(cityId, context),
                    mfcb.newCallback(),
                    query,
                    input);
            }

            mfcb.done();
        }
    }

    private static class SingleCityContext
        implements UniversalSearchProxyRequestContext
    {
        private final Long cityId;
        private final User user;
        private final BasicDistrictSearchContext context;

        public SingleCityContext(
            final Long cityId,
            final BasicDistrictSearchContext context)
        {
            this.cityId = cityId;
            this.user = new User(DistrictConstants.DISTRICT_QUEUE, new LongPrefix(cityId));
            this.context = context;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return context.minPos();
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return context.client();
        }

        @Override
        public Logger logger() {
            return context.logger();
        }

        @Override
        public long lagTolerance() {
            return context.lagTolerance();
        }
    }

    private static class MultiCityMergeCallback
        extends AbstractFilterFutureCallback<List<List<JsonObject>>, List<JsonObject>>
    {
        public MultiCityMergeCallback(
            final FutureCallback<? super List<JsonObject>> callback)
        {
            super(callback);
        }

        @Override
        public void completed(final List<List<JsonObject>> lists) {
            if (lists.size() == 0) {
                callback.completed(Collections.emptyList());
            } else {
                List<JsonObject> result =
                    new ArrayList<>(lists.size() * lists.get(0).size());
                for (List<JsonObject> list: lists) {
                    result.addAll(list);
                }

                callback.completed(result);
            }
        }
    }

    private void executeSingleCity(
        final BasicDistrictSearchContext context,
        final UniversalSearchProxyRequestContext universalContext,
        final FutureCallback<List<JsonObject>> callback,
        final QueryConstructor query,
        final T input)
        throws BadRequestException
    {
        StringBuilder querySb = query.sb();

        MultiFutureCallback<JsonObject> mfcb =
            new MultiFutureCallback<>(callback);

        StringBuilder text = text(context);
        SearchRequestText fuzzyText = SearchRequestText.parseSuggest(
            input.request(),
            context.session().params().getLocale(LOCALE, Locale.ROOT));

        SearchRequestText nonFuzzyText = new SearchRequestText(input.request());

        query.append(PREFIX, universalContext.user().prefix().toStringFast());
        query.append(SERVICE, DistrictConstants.DISTRICT_CITY_QUEUE);

        if (!context.districts().isEmpty()) {
            if (text.length() > 0) {
                text.append(AND);
            }
            text.append(DistrictFields.DISTRICT_ID.prefixedField());
            text.append(":(");
            for (Long dId: context.districts()) {
                text.append(dId);
                text.append(OR);
            }

            text.setLength(text.length() - OR.length());
            text.append(")");
        }

        Rank rank = context.rank();
        if (fuzzyText.hasWords() || nonFuzzyText.hasWords()) {
            if (text.length() > 0) {
                text.append(AND);
            }
            text.append('(');
            text.append('(');
            if (fuzzyText.hasWords()) {
                String fuzzyField = DistrictFields.TEXT.prefixedField();
                fuzzyText.fieldsQuery(text, fuzzyField);
                fuzzyText.negationsQuery(text, fuzzyField);

                if (nonFuzzyText.hasWords()) {
                    text.append(')');
                    text.append(OR);
                    text.append('(');
                }

                if (fuzzyText.text().length() <= SORT_BY_RANK_THRESHOLD) {
                    rank = Rank.DATE;
                }
            }

            if (nonFuzzyText.hasWords()) {
                String wsField = DistrictFields.TEXT_WS.prefixedField();
                nonFuzzyText.fieldsQuery(text, wsField);
                nonFuzzyText.negationsQuery(text, wsField);

                if (nonFuzzyText.text().length() <= SORT_BY_RANK_THRESHOLD) {
                    rank = Rank.DATE;
                }
            }

            text.append(')');
            text.append(')');
        } else {
            rank = Rank.DATE;
            if (text.length() <= 0) {
                text.append(DistrictFields.INDEX_TYPE.prefixedField());
                text.append(":city");
            }
        }

        if (!context.mentionUsers().isEmpty()) {
            if (text.length() > 0) {
                text.append(AND);
            }

            text.append(DistrictFields.MENTION_USERS.field());
            text.append(":(");
            for (Long user: context.mentionUsers()) {
                text.append(user);
                text.append(' ');
            }

            text.setLength(text.length() - 1);
            text.append(')');
        }

        query.append(TEXT, text.toString());
        int baseLength = query.sb().length();

        applyRank(query, context, rank);
        if (context.rank() == Rank.RELEVANCE && rank == Rank.DATE) {
            query.append(LENGTH, context.length());
            context.session().logger().info(
                "Request to small, forcing date order "
                    + fuzzyText.text() + ' ' + nonFuzzyText.text());
        } else {
            query.append(
                LENGTH,
                Math.min(
                    PRUINNING_LENGTH_MULTIPLIER * context.length(),
                    PRUINNING_LENGTH_THRESHOLD));
        }

        if (context.hasTimeRange()) {
            query.append(
                "postfilter",
                "created_at >= " + context.timeRange().min());
            if (context.timeRange().max() < Integer.MAX_VALUE) {
                query.append(
                    "postfilter",
                    "created_at <= " + context.timeRange().max());
            }
        }

        String uri = query.toString();
        query.sb().setLength(baseLength);
        context.searchContext().logger().info(uri);
        FutureCallback<JsonObject> searchCallback = mfcb.newCallback();
        if (rank == Rank.RELEVANCE) {
            CitySearchCallback cityCallback =
                new CitySearchCallback(searchCallback);
            timer.schedule(
                new CitySortFailover(
                    cityCallback,
                    query,
                    context),
                400L);
            searchCallback = cityCallback;
        }

//        ProducerClient client = context.proxy().producerClient().adjust(context.session().context());
//
//        client.executeAllWithInfo(
//            context.user(),
//            context.session().listener().createContextGeneratorFor(client),
//            new DeadlineSearchFailoverCallback<>(
//                context.session(),
//                context,
//                new BasicAsyncRequestProducerGenerator(uri),
//                context.proxy().config().searchFailOverDelay(),
//                deadline,
//                true,
//                JsonAsyncTypesafeDomConsumerFactory.OK,
//                context.contextGenerator(),
//                searchCallback));
        long failover = context.proxy().config().searchFailOverDelay();
        if ((context.request() == null || context.request().isEmpty()) && !context.tags().isEmpty()) {
            failover = 100;
        }

        context.proxy().sequentialRequest(
            context.session(),
            universalContext,
            new BasicAsyncRequestProducerGenerator(uri),
            failover,
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            searchCallback);

        mfcb.done();
    }

    private static class CitySearchCallback extends FilterFutureCallback<JsonObject> {
        private final AtomicBoolean completed = new AtomicBoolean(false);

        public CitySearchCallback(final FutureCallback<? super JsonObject> callback) {
            super(callback);
        }

        public boolean completed() {
            return completed.get();
        }

        @Override
        public void completed(final JsonObject obj) {
            if (completed.compareAndSet(false, true)) {
                callback.completed(obj);
            }
        }

        @Override
        public void failed(final Exception e) {
            if (completed.compareAndSet(false, true)) {
                callback.failed(e);
            }
        }

        @Override
        public void cancelled() {
            if (completed.compareAndSet(false, true)) {
                callback.cancelled();
            }
        }
    }

    private static class CitySortFailover extends TimerTask {
        private final String fallbackUri;
        private final CitySearchCallback callback;
        private final BasicDistrictSearchContext context;

        public CitySortFailover(
            final CitySearchCallback callback,
            final QueryConstructor query,
            final BasicDistrictSearchContext context)
            throws BadRequestException
        {
            this.context = context;
            this.callback = callback;

            query.append("sort", "created_at");
            query.append("collector", "pruning");
            query.append("get", DEFAULT_GET_FIELDS);
            query.append(
                LENGTH,
                Math.min(
                    PRUINNING_LENGTH_MULTIPLIER * context.length(),
                    PRUINNING_LENGTH_THRESHOLD));
            fallbackUri = query.toString();
        }

        @Override
        public void run() {
            if (!callback.completed()) {
                context.logger().info("Trying with prunning: " + fallbackUri);
                context.proxy().sequentialRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(fallbackUri),
                    context.proxy().config().searchFailOverDelay(),
                    true,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    callback);
            }
        }
    }

    // CSON: ParameterNumber

    private StringBuilder text(
        final BasicDistrictSearchContext context)
    {
        StringBuilder sb = new StringBuilder();

        if (context.scope() != DistrictEntityType.ALL) {
            addFilter(
                sb,
                DistrictFields.ENTITY_TYPE,
                context.searchScope(),
                context.scope().entityType());
        }

        String eventType =
            context.session().params().getString("event_type", null);

        if (eventType != null) {
            addFilter(
                sb,
                DistrictFields.EVENT_TYPE,
                context.searchScope(),
                eventType);
        }

        if (!context.tags().isEmpty()) {
            addFilter(
                sb,
                DistrictFields.TAGS,
                context.searchScope(),
                context.tags(),
                AND);
        }

        if (context.districtUser() != null) {
            addFilter(
                sb,
                DistrictFields.USER_ID,
                context.searchScope(),
                context.districtUser());
        }

        if (context.hasTimeRange()) {
            applyTimerange(context, sb, context.searchScope());
        }

        return sb;
    }

    private void applyTimerange(
        final BasicDistrictSearchContext context,
        final StringBuilder uri,
        final SearchScope scope)
    {
        if (scope == SearchScope.DISTRICTS) {
            StringBuilder sb = new StringBuilder("[");
            sb.append(context.timeRange().min());
            sb.append(" TO ");
            sb.append(context.timeRange().max());
            sb.append(']');

            addFilter(uri, DistrictFields.CREATE_DATE, scope, sb.toString());
            return;
        }

        if (uri.length() > 0) {
            uri.append(AND);
        }

        uri.append("create_day_p:");
        int minDay = context.timeRange().min() / 86400 - 1;
        int maxDay = context.timeRange().max() / 86400 + 1;
        if (maxDay - minDay <= 35){
            uri.append('(');
            for (int i = minDay; i<=maxDay; i++) {
                uri.append(i * 86400);
                uri.append(' ');
            }

            uri.setLength(uri.length() - 1);
            uri.append(')');
        } else {
            uri.append('[');
            uri.append(context.timeRange().min());
            uri.append(" TO ");
            uri.append(context.timeRange().max());
            uri.append(']');
        }
    }

    // CSOFF: MultipleStringLiterals
    private void applyRank(
        final QueryConstructor query,
        final BasicDistrictSearchContext context,
        final Rank rank)
        throws BadRequestException
    {
        Collection<String> searchFields =
            SEARCH_FIELDS.get(context.searchScope());
        StringBuilder getFields = new StringBuilder(DEFAULT_GET_FIELDS);

        if (rank == Rank.RELEVANCE) {
            long curDate =
                TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());

            query.append("scorer", PERFIELD.get(context.searchScope()));

            for (String field: searchFields) {
                getFields.append(",#");
                getFields.append(field);
                getFields.append("_non_exact_hits,#");
                getFields.append(field);
                getFields.append(EXACT_POSTFIX);
            }

            getFields.append(",rank");
            StringBuilder sortSb = new StringBuilder();
            if (searchFields.size() > 0) {
                sortSb.append("multi(");
                for (String field: searchFields) {
                    sortSb.append('#');
                    sortSb.append(field);
                    sortSb.append("_exact_hits,");
                }
                sortSb.append("rank)");
            } else {
                sortSb.append("rank");
            }

            if (context.get() != null) {
                for (String field: context.get()) {
                    getFields.append(',');
                    getFields.append(field);
                }
            }

            query.append("sort", sortSb.toString());

            // if you change here, also change rank func in DistrictResultItem
            query.append("dp", "const(0 zero_field)");
            query.append("dp", "fallback(views_cnt,zero_field views_cnt_nn)");
            query.append(
                "dp",
                "fallback(comments_cnt,zero_field comments_cnt_nn)");
            query.append("dp", "fallback(likes_cnt,zero_field likes_cnt_nn)");
            query.append(
                "dp",
                "fallback(dislikes_cnt,zero_field dislikes_cnt_nn)");
            query.append("dp", "const(" + curDate + " cur_date)");
            query.append("dp", "sub(cur_date,created_at secs_passed)");
            query.append("dp", "cdiv(secs_passed,60 minutes_passed)");
            query.append("dp", "csum(minutes_passed,1 age_minutes)");
            query.append("dp", "clog(age_minutes,E time_score)");
            query.append("dp", "csum(views_cnt_nn,1 views_cnt_gz)");
            query.append(
                "dp",
                "fdiv(comments_cnt_nn,views_cnt_gz comments_score)");
            query.append("dp", "fdiv(likes_cnt_nn,views_cnt_gz likes_score)");
            query.append(
                "dp",
                "fdiv(dislikes_cnt_nn,views_cnt_gz dislikes_score)");
            query.append(
                "dp",
                "poly(time_score,-1,comments_score,100,"
                    + "likes_score,50,dislikes_score,-60 rank)");
        } else {
            query.append("sort", "created_at");
            query.append("collector", "pruning(create_day_p)");
        }

        query.append("get", getFields.toString());
    }
    // CSON: MultipleStringLiterals

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super DistrictSearchResult> callback)
        throws HttpException
    {
        BasicDistrictSearchContext context = input.searchContext();

        QueryConstructor query =
            new QueryConstructor(
                "/search-district?IO_PRIO=0&json-type=dollar");

        if (context.scope() == DistrictEntityType.EVENT) {
            //deduplicate multi districts post
            query.append(
                "outer",
                "deduplicate:" + DistrictFields.ENTITY_ID.field());
        }

        AggregateCallback aggregateCallback =
            new AggregateCallback(callback, context, input.request());

        if (context.searchScope() == SearchScope.CITY) {
            executeCity(context, aggregateCallback, query, input);
        } else {
            executeDistricts(context, aggregateCallback, query, input);
        }
    }
    // CSON: MultipleStringLiterals

    private static final class AggregateCallback
        extends AbstractFilterFutureCallback<List<JsonObject>,
        DistrictSearchResult>
    {
        private final BasicDistrictSearchContext context;
        private final long fetchTime;
        private final List<String> preparedRequest;

        private AggregateCallback(
            final FutureCallback<? super DistrictSearchResult> callback,
            final BasicDistrictSearchContext context,
            final String request)
        {
            super(callback);
            this.context = context;

            this.fetchTime =
                TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());

            this.preparedRequest =
                IntervalHighlighter.INTANCE.prepareRequest(request);
        }

        @Override
        public void completed(final List<JsonObject> lists) {
            List<String> sortFields;
            if (context.rank() == Rank.RELEVANCE) {
                sortFields = SORT_FIELDS.get(context.searchScope());
            } else {
                sortFields = Collections.emptyList();
            }

            DistrictSearchResult result = new DistrictSearchResult();
            long total = 0L;
            try {
                for (JsonObject response: lists) {
                    total += response.asMap().getLong("hitsCount");
                    JsonList hits = response.asMap().get("hitsArray").asList();
                    for (JsonObject hit: hits) {
                        JsonMap doc = hit.asMap();
                        String text =
                            doc.getOrNull(DistrictFields.TEXT.field());

                        List<IntInterval> highlights = Collections.emptyList();
                        if (text != null) {
                            try {
                                highlights =
                                    IntervalHighlighter.INTANCE
                                        .highlightIntervals(
                                        text,
                                        preparedRequest,
                                        true,
                                        true);
                            } catch (Exception e) {
                                StringBuilder sb =
                                    new StringBuilder("Bad highlight for ");
                                sb.append("request ");
                                sb.append(text);
                                sb.append(" ");
                                sb.append(
                                    JsonType.HUMAN_READABLE.toString(doc));
                                context.logger().log(
                                    Level.WARNING,
                                    sb.toString(),
                                    e);
                            }
                        }
                        if (!result.add(
                            new DistrictResultItem(
                                doc,
                                highlights,
                                sortFields,
                                fetchTime)))
                        {
                            total -= 1;
                        }
                    }

                    result.addTotal(total);
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            callback.completed(result);
        }
    }
}
