package ru.yandex.search.district.search;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.LinkedHashSet;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.tukaani.xz.LZMA2Options;
import org.tukaani.xz.LZMAInputStream;
import org.tukaani.xz.LZMAOutputStream;

import ru.yandex.base64.Base64;
import ru.yandex.base64.Base64Decoder;
import ru.yandex.base64.Base64Encoder;
import ru.yandex.function.ByteArrayVoidProcessor;

public class DistrictSearchSession
    implements ByteArrayVoidProcessor<IOException>
{
    private static final LZMA2Options OPTIONS = new LZMA2Options();
    private Set<Long> ids;
    private final int offset;

    public DistrictSearchSession() {
        this.offset = 0;
        this.ids = new LinkedHashSet<>();
    }

    public DistrictSearchSession(
        final String sessionIdRaw,
        final int offset)
        throws IOException
    {
        this.offset = offset;

        if (sessionIdRaw != null && !sessionIdRaw.isEmpty()) {
            Base64Decoder b64decoder = new Base64Decoder(Base64.URL);
            b64decoder.process(sessionIdRaw.toCharArray());
            b64decoder.processWith(this);
        } else {
            this.ids = new LinkedHashSet<>();
        }
    }

    @Override
    public void process(
        final byte[] buf,
        final int off,
        final int len)
        throws IOException
    {
        LZMAInputStream is =
            new LZMAInputStream(new ByteArrayInputStream(buf, off, len));

        byte[] bytes = IOUtils.toByteArray(is);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        Set<Long> ids =
            new LinkedHashSet<>(bytes.length / Long.BYTES);

        int left = offset;
        while (buffer.hasRemaining() && left > 0) {
            ids.add(buffer.getLong());
            left -= 1;
        }

        this.ids = ids;
    }

    public Set<Long> getDocs() {
        return ids;
    }

    /**
     * Here the point, we keep in session up to offset elements
     * after init
     * @param item
     * @return true if passed
     */
    public boolean add(final DistrictResultItem item) {
        boolean addRes = ids.add(item.entityId());
        if (ids.size() < offset) {
            return false;
        }

        return addRes;
    }

    public String getSessionId() throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(ids.size());

        ByteBuffer buffer =
            ByteBuffer.allocate(ids.size() * Long.BYTES);
        for (Long id: ids) {
            buffer.putLong(id);
        }

        byte[] data = buffer.array();

        LZMAOutputStream os = new LZMAOutputStream(bos, OPTIONS, data.length);
        os.write(data, 0, data.length);
        os.close();

        Base64Encoder b64Encoder = new Base64Encoder(Base64.URL);
        b64Encoder.process(bos.toByteArray());

        return b64Encoder.toStringFast();
    }
}
