package ru.yandex.search.district.suggest;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.LongParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.district.DistrictConstants;
import ru.yandex.search.district.DistrictPopularRequestsFields;
import ru.yandex.search.district.DistrictSearchProxy;
import ru.yandex.search.district.search.DistrictSearchContext;
import ru.yandex.search.district.search.SearchScope;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class BasicDistrictSuggestContext
    implements DistrictSearchContext,
    RequestProvider,
    DistrictSuggestContextProvider
{
    private static final Long DEFAULT_FAILOVER = 50L;
    private static final int DEFAULT_LENGTH = 10;
    private static final int DEFAULT_MIN_REQUESTS = 10;
    private static final User USER =
        new User(
            DistrictConstants.DISTRICT_CITY_QUEUE,
            new LongPrefix(DistrictPopularRequestsFields.prefix()));

    private static final CollectionParser<Long, Set<Long>, Exception>
        DS_PARSER =
        new CollectionParser<>(
            LongParser.INSTANCE,
            LinkedHashSet::new);

    private static final CollectionParser<String, Set<String>, Exception>
        STR_SET_PARSER =
        new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            LinkedHashSet::new);

    private final ProxySession session;
    private final JsonType jsonType;
    private final AsyncClient client;
    private final Supplier<? extends HttpClientContext> contextGenerator;
    private final String request;
    private final DistrictSearchProxy proxy;
    private final Collection<Long> districts;
    private final SearchScope searchScope;
    private final int length;
    private final int minRequestsCount;
    private final boolean allowLaggingHosts;

    private final Long cityId;

    private final String user;
    private final Collection<String> get;

    public BasicDistrictSuggestContext(
        final DistrictSearchProxy proxy,
        final ProxySession session)
        throws BadRequestException
    {
        this.proxy = proxy;
        this.session = session;

        CgiParams params = session.params();
        Collection<Long> districts =
            params.get(
                "districts",
                Collections.emptySet(),
                DS_PARSER);
        if (districts.isEmpty()) {
            districts =
                params.getAll("district", Collections.emptySet(), DS_PARSER);
        }

        get = params.get("get", null, STR_SET_PARSER);

        this.cityId = params.getLong("city", null);

        if (cityId == null) {
            searchScope = SearchScope.DISTRICTS;
        } else if (districts.isEmpty()) {
            searchScope = SearchScope.CITY;
        } else {
            searchScope = null;
        }

        this.districts = districts;

        length = params.get(
            "length",
            DEFAULT_LENGTH,
            PositiveIntegerValidator.INSTANCE);

        minRequestsCount =
            params.getInt("min-requests", DEFAULT_MIN_REQUESTS);

        user = params.getString("user", null);

        allowLaggingHosts =
            params.getBoolean("allow-lagging-hosts", false);

        request = params.getString("request");

        client = proxy.searchClient().adjust(session.context());
        contextGenerator =
            session.listener().createContextGeneratorFor(client);
        jsonType = JsonTypeExtractor.NORMAL.extract(params);
    }

    public Long backendFailoverDelay() {
        return DEFAULT_FAILOVER;
    }

    public Collection<String> get() {
        return get;
    }

    @Override
    public User user() {
        return USER;
    }

    public String districtUser() {
        return user;
    }

    @Override
    public Long minPos() {
        return null;
    }

    @Override
    public AbstractAsyncClient<?> client() {
        return client;
    }

    @Override
    public Logger logger() {
        return session.logger();
    }

    @Override
    public long lagTolerance() {
        return allowLaggingHosts ? Long.MAX_VALUE : 0L;
    }

    public ProxySession session() {
        return session;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public Supplier<? extends HttpClientContext> contextGenerator() {
        return contextGenerator;
    }

    @Override
    public String request() {
        return request;
    }

    public DistrictSearchProxy proxy() {
        return proxy;
    }

    public Collection<Long> districts() {
        return districts;
    }

    public int length() {
        return length;
    }

    @Override
    public BasicDistrictSuggestContext suggestContext() {
        return this;
    }

    public Long cityId() {
        return cityId;
    }

    public SearchScope searchScope() {
        return searchScope;
    }

    public int minRequestsCount() {
        return minRequestsCount;
    }
}
