package ru.yandex.search.district.suggest;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpException;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.search.district.DistrictSearchProxy;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.ConcatResultsRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.TranslitRule;

public class DistrictSuggestHandler implements ProxyRequestHandler {
    private static final long MINUTES_EXPIRE = 10;
    private static final long MAX_SIZE = 500000;

    private final Cache<String, DistrictSuggestItem[]> cache;

    private final SearchRule<BasicDistrictSuggestContext,
        List<DistrictSuggestItem>> rule;

    private final DistrictSearchProxy proxy;

    public DistrictSuggestHandler(final DistrictSearchProxy proxy) {
        this.proxy = proxy;
        this.rule = new TranslitRule<>(
            new ChainedSearchRule<>(
                new ConcatResultsRule<>(
                    new ChainedSearchRule<>(
                        new PopularRequestsSuggestRule<>(),
                        (input, request) ->
                            new DistrictSuggestContextRequestProvider(
                                input.suggestContext(),
                                request))),
                (input, requests) ->
                    new DistrictSuggestContextRequestsProvider(
                        input.suggestContext(),
                        requests)),
            true);

        this.cache =
            CacheBuilder.newBuilder()
                .expireAfterWrite(MINUTES_EXPIRE, TimeUnit.MINUTES)
                .maximumSize(MAX_SIZE)
                .concurrencyLevel(proxy.config().workers()).build();
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        BasicDistrictSuggestContext context =
            new BasicDistrictSuggestContext(proxy, session);

        String normalized =
            SearchRequestText.normalizeSuggest(context.request());

        DistrictSuggestPrinter printer =
            new DistrictSuggestPrinter(context, normalized, cache);

        if (printer.isRequestCachable()) {
            DistrictSuggestItem[] cached = cache.getIfPresent(normalized);
            if (cached != null) {
                printer.completedCached(cached);
                return;
            }
        }

        rule.execute(context, printer);
    }
}
