package ru.yandex.search.district.suggest;

import java.io.IOException;
import java.util.Comparator;
import java.util.List;

import com.google.common.cache.Cache;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.collection.IntInterval;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;

public class DistrictSuggestPrinter
    extends AbstractProxySessionCallback<List<DistrictSuggestItem>>
{
    private static final Comparator<? super DistrictSuggestItem> CMP
        = Comparator.comparingInt(DistrictSuggestItem::count).reversed();
    private static final int MAX_LETTER_CACHE = 3;
    private final BasicDistrictSuggestContext context;
    private final String normalized;
    private final boolean cacheable;
    private final Cache<String, DistrictSuggestItem[]> cache;

    public DistrictSuggestPrinter(
        final BasicDistrictSuggestContext context,
        final String normalized,
        final Cache<String, DistrictSuggestItem[]> cache)
    {
        super(context.session());
        this.context = context;
        this.normalized = normalized;
        this.cacheable = calcCacheable();
        this.cache = cache;
    }

    private boolean calcCacheable() {
        return normalized.length() <= MAX_LETTER_CACHE;
    }

    public boolean isRequestCachable() {
        return cacheable;
    }

    protected void completedCached(final DistrictSuggestItem[] items) {
        StringBuilderWriter sbw = new StringBuilderWriter();

        int max =
            Math.min(context.length(), items.length);

        try (JsonWriter writer = context.jsonType().create(sbw)) {
            writer.startArray();
            for (int i = 0; i < max; i++) {
                writer.startObject();
                writer.key("highlight");
                writer.startArray();
                IntInterval[] intervals = items[i].highlight();
                for (IntInterval interval: intervals) {
                    writer.startArray();
                    writer.value(interval.min());
                    writer.value(interval.max());
                    writer.endArray();
                }
                writer.endArray();
                writer.key("text");
                writer.value(items[i].request());
                writer.endObject();
            }
            writer.endArray();
        } catch (IOException ioe) {
            failed(ioe);
            return;
        }

        session.response(
            HttpStatus.SC_OK,
            new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(context.session().acceptedCharset())));
    }

    private int length(final int suggests) {
        return Math.min(context.length(), suggests);
    }

    @Override
    public void completed(final List<DistrictSuggestItem> items) {
        items.sort(CMP);
        int max = length(items.size());

        DistrictSuggestItem[] cached = new DistrictSuggestItem[max];
        for (int i = 0; i < max; i++) {
            cached[i] = items.get(i);
        }

        if (cacheable) {
            cache.put(normalized, cached);
        }

        completedCached(cached);
    }
}
