package ru.yandex.search.district.suggest;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.IntInterval;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.district.DistrictPopularRequestsFields;
import ru.yandex.search.ps.highlight.IntervalHighlighter;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class PopularRequestsSuggestRule<T extends RequestProvider
    & DistrictSuggestContextProvider>
    implements SearchRule<T, List<DistrictSuggestItem>>
{
    private static final String POSTFILTER =
        DistrictPopularRequestsFields.REQUEST_COUNT.fieldName()
            + " >= ";
    private static final String GET =
        DistrictPopularRequestsFields.REQUEST_TEXT.fieldName()
            + ','
            + DistrictPopularRequestsFields.REQUEST_COUNT.fieldName();

    public PopularRequestsSuggestRule() {
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<DistrictSuggestItem>> callback)
        throws HttpException
    {
        BasicDistrictSuggestContext context = input.suggestContext();
        SearchRequestText request = SearchRequestText.parseSuggest(
            input.request(),
            context.session().params().getLocale("locale", Locale.ROOT));
        StringBuilder sb = new StringBuilder();
        if (request.hasWords()) {
            // require at least one token present in file name
            request.fieldsQuery(
                sb,
                DistrictPopularRequestsFields.REQUEST_TEXT.fieldName());
        } else {
            sb.append(DistrictPopularRequestsFields.REQUEST_TEXT.fieldName());
            sb.append(':');
            sb.append('*');
        }

        QueryConstructor query =
            new QueryConstructor("/search-popular-suggest?IO_PRIO=0");
        query.append(
            "postfilter",
            POSTFILTER + context.minRequestsCount());
        query.append(
            "sort",
            DistrictPopularRequestsFields.REQUEST_COUNT.fieldName());
        query.append("get", GET);
        query.append("length", context.length());
        query.append("service", context.user().service());
        query.append("prefix", context.user().prefix().toStringFast());
        query.append("text", sb.toString());

        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.backendFailoverDelay(),
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new ErrorSuppressingFutureCallback<>(
                new Callback(callback, context, input.request()),
                x -> RequestErrorType.HTTP,
                JsonMap.EMPTY));
    }

    private static final class Callback
        extends AbstractFilterFutureCallback<JsonObject,
        List<DistrictSuggestItem>>
    {
        private final String request;
        private final BasicDistrictSuggestContext context;

        private Callback(
            final FutureCallback<? super List<DistrictSuggestItem>> callback,
            final BasicDistrictSuggestContext context,
            final String request)
        {
            super(callback);

            this.request = request;
            this.context = context;
        }

        @Override
        public void completed(final JsonObject luceneResult) {
            try {
                List<String> prepared =
                    IntervalHighlighter.INTANCE.prepareRequest(request);
                JsonMap response = luceneResult.asMap();
                JsonList hits = response.get("hitsArray").asList();
                List<DistrictSuggestItem> items = new ArrayList<>(hits.size());
                for (JsonObject hit: hits) {
                    JsonMap doc = hit.asMap();
                    int count =
                        doc.getInt(
                            DistrictPopularRequestsFields.REQUEST_COUNT
                                .fieldName(),
                            0);
                    String request =
                        doc.getOrNull(
                            DistrictPopularRequestsFields.REQUEST_TEXT
                                .fieldName());

                    if (request != null) {
                        List<IntInterval> highlight;
                        try {
                            highlight =
                                IntervalHighlighter.INTANCE.highlightIntervals(
                                    request,
                                    prepared,
                                    true,
                                    true);
                        } catch (Exception e) {
                            context.logger().log(
                                Level.WARNING,
                                "Failed to highlight "
                                    + request + ' ' + prepared,
                                e);
                            highlight = Collections.emptyList();
                        }

                        items.add(
                            new DistrictSuggestItem(request, highlight, count));
                    }
                }

                context.session().connection().setSessionInfo(
                    SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
                    Long.toString(items.size()));

                callback.completed(items);
            } catch (JsonException e) {
                callback.completed(Collections.emptyList());
                //failed(new ServiceUnavailableException(e));
            }
        }
    }
}
