package ru.yandex.search.district.wizard;

import java.io.IOException;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.district.DistrictConstants;
import ru.yandex.search.district.DistrictEntityType;
import ru.yandex.search.district.DistrictFields;
import ru.yandex.search.district.DistrictSearchProxy;
import ru.yandex.search.district.search.DistrictSearchContext;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class DistrictWizardSearchContext
    implements DistrictSearchContext,
    RequestProvider,
    DistrictWizardContextProvider
{
    private static final int DEFAULT_LENGTH = 1;

    private static final String DEFAULT_GET =
        DistrictFields.ID.field() + ','
            + DistrictFields.TEXT.field() + ','
            + DistrictFields.ENTITY_ID.field();

    private final int length;
    private final String request;
    private final User user;
    private final DistrictSearchProxy proxy;
    private final ProxySession session;
    private final JsonType jsonType;
    private final AsyncClient client;
    private final String get;
    private final Long cityId;
    private final boolean onlyNice;
    private final Supplier<? extends HttpClientContext> contextGenerator;
    private final DistrictEntityType entityType;

    public DistrictWizardSearchContext(
        final DistrictSearchProxy proxy,
        final ProxySession session)
        throws BadRequestException, IOException
    {
        this.proxy = proxy;
        this.session = session;

        CgiParams params = session.params();
        this.cityId = params.getLong("city", null);
        this.entityType =
            params.getEnum(
                DistrictEntityType.class,
                "scope",
                DistrictEntityType.EVENT);

        this.length = params.get(
            "length",
            DEFAULT_LENGTH,
            PositiveIntegerValidator.INSTANCE);

        jsonType = JsonTypeExtractor.NORMAL.extract(params);

        request = params.getString("request");

        get = params.getString("get", DEFAULT_GET);

        onlyNice = params.getBoolean("nice_only", true);

        user = new CustomUser(
            DistrictConstants.DISTRICT_CITY_QUEUE,
            new LongPrefix(cityId),
            request);

        client = proxy.wizardSerchClient().adjust(session.context());
        contextGenerator =
            session.listener().createContextGeneratorFor(client);
    }

    @Override
    public DistrictSearchProxy proxy() {
        return proxy;
    }

    @Override
    public ProxySession session() {
        return session;
    }

    @Override
    public Supplier<? extends HttpClientContext> contextGenerator() {
        return contextGenerator;
    }

    @Override
    public int length() {
        return length;
    }

    @Override
    public JsonType jsonType() {
        return jsonType;
    }

    @Override
    public User user() {
        return user;
    }

    @Override
    public Long minPos() {
        return null;
    }

    @Override
    public AbstractAsyncClient<?> client() {
        return client;
    }

    @Override
    public Logger logger() {
        return session.logger();
    }

    @Override
    public long lagTolerance() {
        return Long.MAX_VALUE;
    }

    public Long cityId() {
        return cityId;
    }

    public String request() {
        return request;
    }

    public String get() {
        return get;
    }

    public DistrictEntityType entityType() {
        return entityType;
    }

    @Override
    public DistrictWizardSearchContext searchContext() {
        return this;
    }

    public boolean onlyNice() {
        return onlyNice;
    }

    private static final class CustomUser extends User {
        private final String request;

        private CustomUser(
            final String service,
            final Prefix prefix,
            final String request)
        {
            super(service, prefix);

            if (request == null) {
                this.request = "";
            } else {
                this.request = request.trim();
            }
        }

        @Override
        public boolean equals(final Object o) {
            return super.equals(o);
        }

        @Override
        public int hashCode() {
            return 2 * super.hashCode() + request.hashCode();
        }
    }
}
