package ru.yandex.erratum;

import java.nio.charset.CharacterCodingException;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.function.StringVoidProcessor;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;

public class ErratumClient extends AbstractAsyncClient<ErratumClient> {
    private static final int PORNO_FLAG = 0x40;
    private static final int NO_TRANSLIT_FLAG = 0x80;
    private static final int IGNORE_NICKNAMES_FLAG = 0x100;
    private static final int NO_KEYBOARD_FLAG = 0x400;

    private final StringVoidProcessor<char[], CharacterCodingException>
        encoder =
            new StringVoidProcessor<>(new PctEncoder(PctEncodingRule.QUERY));
    private final HttpHost host;
    private final String uri;

    public ErratumClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableErratumConfig erratumConfig)
    {
        super(reactor, erratumConfig);
        host = erratumConfig.host();
        int options = 1;
        if (erratumConfig.enablePorno()) {
            options |= PORNO_FLAG;
        }
        if (!erratumConfig.enableTranslit()) {
            options |= NO_TRANSLIT_FLAG;
        }
        if (!erratumConfig.enableKeyboard()) {
            options |= NO_KEYBOARD_FLAG;
        }
        if (erratumConfig.ignoreNicknames()) {
            options |= IGNORE_NICKNAMES_FLAG;
        }
        StringBuilder sb = new StringBuilder("/misspell.json/check?srv=");
        sb.append(erratumConfig.service());
        sb.append("&options=");
        sb.append(options);
        sb.append("&text=");
        uri = sb.toString();
    }

    protected ErratumClient(
        final CloseableHttpAsyncClient client,
        final ErratumClient sample)
    {
        super(client, sample);
        host = sample.host;
        uri = sample.uri;
    }

    @Override
    protected ErratumClient adjust(final CloseableHttpAsyncClient client) {
        return new ErratumClient(client, this);
    }

    public Future<ErratumResult> execute(final String text)
        throws CharacterCodingException
    {
        return execute(text, EmptyFutureCallback.INSTANCE);
    }

    public Future<ErratumResult> execute(
        final String text,
        final FutureCallback<? super ErratumResult> callback)
        throws CharacterCodingException
    {
        return execute(text, httpClientContextGenerator(), callback);
    }

    public Future<ErratumResult> execute(
        final String text,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super ErratumResult> callback)
        throws CharacterCodingException
    {
        StringBuilder sb = new StringBuilder(uri);
        StringBuilderProcessorAdapter adapter =
            new StringBuilderProcessorAdapter(sb);
        synchronized (encoder) {
            encoder.process(text);
            encoder.processWith(adapter);
        }
        return execute(
            host,
            new BasicAsyncRequestProducerGenerator(new String(sb)),
            ErratumConsumerFactory.OK,
            contextGenerator,
            callback);
    }
}

