package com.github.jelmerk.knn.util;

import java.io.Serializable;
import java.util.Arrays;

/**
 * Bitset.
 */
public class ArrayBitSet implements BitSet, Serializable {

    private static final long serialVersionUID = 1L;

    private final int[] buffer;
    private final int count;

    /**
     * Initializes a new instance of the {@link ArrayBitSet} class.
     *
     * @param count The number of items in the set.
     */
    public ArrayBitSet(int count) {
        this.count = count;
        this.buffer = new int[(count >> 5) + 1];
    }

    private ArrayBitSet(final int count, final int[] buffer) {
        this.count = count;
        this.buffer = buffer;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean contains(int id) {
        int carrier = this.buffer[id >> 5];
        return ((1 << (id & 31)) & carrier) != 0;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void add(int id)  {
        int mask = 1 << (id & 31);
        this.buffer[id >> 5] |= mask;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void remove(int id) {
        int mask = 1 << (id & 31);
        this.buffer[id >> 5] &= ~mask;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void clear() {
        Arrays.fill(this.buffer, 0);
    }

    @Override
    public BitSet clone(final int count) {
        int arraySize = (count >> 5) + 1;
        return new ArrayBitSet(
            count,
            Arrays.copyOf(buffer, arraySize));
    }

    @Override
    public int size() {
        return count;
    }
}
