package com.github.jelmerk.knn.util;

import java.io.Serializable;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.function.Supplier;

/**
 * Generic object pool.
 *
 * @param <T> type of object to pool
 */
public class GenericObjectPool<T> implements Serializable {

    private static final long serialVersionUID = 1L;

    private final ArrayBlockingQueue<T> items;
    private final Supplier<T> supplier;

    /**
     * Constructs a new pool
     *
     * @param supplier used to create instances of the object to pool
     * @param maxPoolSize maximum items to have in the pool
     */
    public GenericObjectPool(Supplier<T> supplier, int maxPoolSize) {
        this.items = new ArrayBlockingQueue<>(maxPoolSize);
        this.supplier = supplier;
    }

    /**
     * Borrows an object from the pool.
     *
     * @return the borrowed object
     */
    public T borrowObject() {
        T t = items.poll();
        if (t == null) {
            t = supplier.get();
        }
        return t;
    }

    /**
     * Returns an instance to the pool. By contract, obj must have been obtained using {@link GenericObjectPool#borrowObject()}
     *
     * @param item the item to return to the pool
     */
    public void returnObject(T item) {
        items.add(item);
    }

}
