package ru.yandex.jzabkv;

import java.io.File;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.LongMemorySizeParser;

public abstract class AbstractJZabKVConfigBuilder
    <T extends AbstractJZabKVConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements JZabKVConfig
{
    private File logDir;
    private File dataDir;
    private String serverId;
    private Set<String> peers;
    private long peerTimeout;
    private long syncTimeout;
    private int bossNioThreads;
    private int workerNioThreads;
    private long deliverSnapshotInterval;
    private long modifySnapshotInterval;
    private long sizeSnapshotInterval;
    private long maxDatabaseSize;
    private long defaultTTL;

    public AbstractJZabKVConfigBuilder(
        final JZabKVConfig config)
        throws ConfigException
    {
        super(config);
        logDir = config.logDir();
        dataDir = config.dataDir();
        serverId = config.serverId();
        peers = Collections.unmodifiableSet(config.peers());
        peerTimeout = config.peerTimeout();
        syncTimeout = config.syncTimeout();
        bossNioThreads = config.bossNioThreads();
        workerNioThreads = config.workerNioThreads();
        deliverSnapshotInterval = config.deliverSnapshotInterval();
        modifySnapshotInterval = config.modifySnapshotInterval();
        sizeSnapshotInterval = config.sizeSnapshotInterval();
        maxDatabaseSize = config.maxDatabaseSize();
        defaultTTL = config.defaultTTL();
    }

    public AbstractJZabKVConfigBuilder(
        final IniConfig config,
        final JZabKVConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        logDir = config.getDir("logdir", defaults.logDir());
        dataDir = config.getDir("datadir", defaults.dataDir());
        serverId = config.getString("server-id", defaults.serverId());
        peers = config.get(
            "peers",
            Collections.emptySet(),
            new CollectionParser<>(String::trim, LinkedHashSet::new, ','));
        peerTimeout = config.get(
            "peer-timeout",
            defaults.peerTimeout(),
            DurationParser.POSITIVE_LONG);
        syncTimeout = config.get(
            "sync-timeout",
            defaults.syncTimeout(),
            DurationParser.POSITIVE_LONG);
        bossNioThreads = config.getInt(
            "boss-nio-threads",
            defaults.bossNioThreads());
        workerNioThreads = config.getInt(
            "worker-nio-threads",
            defaults.workerNioThreads());
        deliverSnapshotInterval = config.getLong(
            "deliver-snapshot-interval",
            defaults.deliverSnapshotInterval());
        modifySnapshotInterval = config.getLong(
            "modify-snapshot-interval",
            defaults.modifySnapshotInterval());
        sizeSnapshotInterval = config.getLong(
            "size-snapshot-interval",
            defaults.sizeSnapshotInterval());
        maxDatabaseSize = config.get(
            "max-database-size",
            defaults.maxDatabaseSize(),
            LongMemorySizeParser.INSTANCE);
        defaultTTL = config.get(
            "default-ttl",
            defaults.defaultTTL(),
            DurationParser.POSITIVE_LONG);
    }

    @Override
    public File logDir() {
        return logDir;
    }

    public T logDir(final File logDir) {
        this.logDir = logDir;
        return self();
    }

    @Override
    public File dataDir() {
        return dataDir;
    }

    public T dataDir(final File dataDir) {
        this.dataDir = dataDir;
        return self();
    }

    @Override
    public String serverId() {
        return serverId;
    }

    public T serverId(final String serverId) {
        this.serverId = serverId;
        return self();
    }

    @Override
    public Set<String> peers() {
        return peers;
    }

    public T peers(final Set<String> peers) {
        this.peers = Collections.unmodifiableSet(peers);
        return self();
    }

    @Override
    public long peerTimeout() {
        return peerTimeout;
    }

    public T peerTimeout(final long peerTimeout) {
        this.peerTimeout = peerTimeout;
        return self();
    }

    @Override
    public long syncTimeout() {
        return syncTimeout;
    }

    public T syncTimeout(final long syncTimeout) {
        this.syncTimeout = syncTimeout;
        return self();
    }

    @Override
    public int bossNioThreads() {
        return bossNioThreads;
    }

    public T bossNioThreads(final int bossNioThreads) {
        this.bossNioThreads = bossNioThreads;
        return self();
    }

    @Override
    public int workerNioThreads() {
        return workerNioThreads;
    }

    public T workerNioThreads(final int workerNioThreads) {
        this.workerNioThreads = workerNioThreads;
        return self();
    }

    @Override
    public long deliverSnapshotInterval() {
        return deliverSnapshotInterval;
    }

    public T deliverSnapshotInterval(final long deliverSnapshotInterval) {
        this.deliverSnapshotInterval = deliverSnapshotInterval;
        return self();
    }

    @Override
    public long modifySnapshotInterval() {
        return modifySnapshotInterval;
    }

    public T modifySnapshotInterval(final long modifySnapshotInterval) {
        this.modifySnapshotInterval = modifySnapshotInterval;
        return self();
    }

    @Override
    public long sizeSnapshotInterval() {
        return sizeSnapshotInterval;
    }

    public T sizeSnapshotInterval(final long sizeSnapshotInterval) {
        this.sizeSnapshotInterval = sizeSnapshotInterval;
        return self();
    }

    @Override
    public long maxDatabaseSize() {
        return maxDatabaseSize;
    }

    public T maxDatabaseSize(final long maxDatabaseSize) {
        this.maxDatabaseSize = maxDatabaseSize;
        return self();
    }

    @Override
    public long defaultTTL() {
        return defaultTTL;
    }

    public T defaultTTL(final long defaultTTL) {
        this.defaultTTL = defaultTTL;
        return self();
    }

    public void copyTo(final AbstractJZabKVConfigBuilder<?> other) {
        copyTo((AbstractHttpProxyConfigBuilder<?>) other);
    }
}
