package ru.yandex.jzabkv;

import com.github.zk1931.jzab.PendingRequests;
import com.github.zk1931.jzab.PendingRequests.Tuple;
import com.github.zk1931.jzab.StateMachine;
import com.github.zk1931.jzab.Zab;
import com.github.zk1931.jzab.ZabConfig;
import com.github.zk1931.jzab.ZabException;
import com.github.zk1931.jzab.Zxid;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import java.nio.ByteBuffer;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;

import java.util.Collection;
import java.util.Set;
//import java.util.concurrent.ConcurrentHashMap;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.jzabkv.command.CleanupCommand;
import ru.yandex.jzabkv.command.Command;
import ru.yandex.jzabkv.key.Key;
import ru.yandex.jzabkv.node.Node;
import ru.yandex.jzabkv.storage.NonBlockingHashMapStorage;
//import ru.yandex.jzabkv.storage.ConcurrentLinkedHashMapStorage;
//import ru.yandex.jzabkv.storage.SimpleStorage;
//import ru.yandex.jzabkv.storage.MVStoreStorage;
//import ru.yandex.jzabkv.storage.RocksDBStorage;
//import ru.yandex.jzabkv.storage.LuceneStorage;
import ru.yandex.jzabkv.storage.Storage;

/**
 * StateMachine of the ZabKV.
 */
public final class Database implements Closeable, StateMachine {
    private static final long CLEANUP_INTERVAL = 100;

    private Zab zab;
//    private String serverId;
    private final ZabConfig zabConfig;
    private final JZabKV server;
    private final Logger logger;
    private final long deliverSnapshotInterval;
    private final long modifySnapshotInterval;
    private final long sizeSnapshotInterval;
    private ReusableDataInput preprocessInput;
    private ReusableDataInput deliverInput;
    private long deliverCount = 0;
    private long modifyCount = 0;
    private long transferredSize = 0;
    private volatile boolean broadcasting = false;
    private volatile boolean isLeader = false;
    private Cleanuper cleanuper;

    private final Storage storage;

    public Database(final JZabKV server) throws IOException {
        this.server = server;
        zabConfig = new ZabConfig();
        deliverSnapshotInterval = server.config().deliverSnapshotInterval();
        modifySnapshotInterval = server.config().modifySnapshotInterval();
        sizeSnapshotInterval = server.config().sizeSnapshotInterval();
        zabConfig.setLogDir(server.config().logDir().getPath());
        zabConfig.setTimeoutMs((int) server.config().peerTimeout());
        zabConfig.setMinSyncTimeoutMs((int) server.config().syncTimeout());
        System.err.println("SyncTimeout: " + zabConfig.getMinSyncTimeoutMs());
        zabConfig.setBossNioThreads(server.config().bossNioThreads());
        zabConfig.setWorkerNioThreads(server.config().workerNioThreads());
        zabConfig.setSnapshotRetainCount(2);
        Set<String> peers = server.config().peers();
        String serverId = server.config().serverId();
        Logger zabLogger = server.logger().replacePrefix("ZAB");

        cleanupSnapshots(server.config().logDir().toPath());
//        this.storage = new LuceneStorage(server.config().dataDir());
//        this.storage = new RocksDBStorage(server.config().dataDir());
//        this.storage = new SimpleStorage(server.config().dataDir());
        this.storage = new NonBlockingHashMapStorage(
            server.config().maxDatabaseSize(),
            server.logger().replacePrefix("NBHMS"));
        if (isDirEmpty(server.config().logDir().toPath())) {
            zab = new Zab(zabLogger, this, zabConfig, serverId, peers);
        } else {
            zab = new Zab(zabLogger, this, zabConfig);
        }
        serverId = zab.getServerId();
        this.logger = server.logger().replacePrefix(serverId);
        cleanuper = new Cleanuper();
        cleanuper.start();
    }

    private void cleanupSnapshots(final Path dir) throws IOException {
        try (DirectoryStream<Path> dirStream =
            Files.newDirectoryStream(dir))
        {
            for (Path path: dirStream) {
                File f = path.toFile();
                if (f.isFile() && f.getName().startsWith("snapshot")) {
                    server.logger().severe("Removing orphaned snapshot: "
                        + f.getName());
                    f.delete();
                }
            }
        }
    }

    private static boolean isDirEmpty(final Path directory) throws IOException {
        try (DirectoryStream<Path> dirStream =
            Files.newDirectoryStream(directory))
        {
            return !dirStream.iterator().hasNext();
        }
    }

    public void snapshot(final FutureCallback<String> callback)
        throws ZabException
    {
        zab.takeSnapshot(callback);
    }

    public void snapshot(final boolean doWait) throws ZabException {
        FutureCallback<String> cb = new FutureCallback<>() {
            @Override
            public void completed(final String path) {
                logger.severe("Implicit shapshot completed: " + path);
                synchronized (this) {
                    notify();
                }
            }

            @Override
            public void failed(final Exception e) {
                logger.log(
                    Level.SEVERE,
                    "Snaphot error",
                    e);
                synchronized (this) {
                    notify();
                }
            }

            @Override
            public void cancelled() {
            }
        };
        zab.takeSnapshot(cb);
        if (doWait) {
            synchronized (cb) {
                try {
                    cb.wait();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

//    public long snapshotInterval() {
//        return snapshotInterval;
//    }
    @Override
    public void close() throws IOException {
        storage.close();
    }

    public void clear() {
        storage.clear();
    }

    public Node get(final Key key) {
        Node node = storage.get(key);
        if (node != null) {
            long currentTime = System.currentTimeMillis();
            if (node.evictionTimestamp() <= currentTime) {
                node = null;
            }
        }
        return node;
    }

    public void put(final Node node) {
//        logger.info("Putting: " + node);
        storage.put(node);
        modifyCount++;
        checkSnapshotInterval(false);
    }

    public Collection<Key> tryPut(final Node node) {
        try {
            Collection<Key> toEvict = storage.tryPut(node);
            if (toEvict.size() > 0) {
                logger.info("Evicting: " + toEvict.size() + " nodes");
                server.nodesEvicted(toEvict.size());
            }
            return toEvict;
        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        }
    }

    public void delete(final Key key) {
        storage.delete(key);
        modifyCount++;
        checkSnapshotInterval(false);
    }

    public void add(
        final Command command,
        final FutureCallback<Void> context)
        throws IOException, ZabException
    {
        ByteBuffer bb = Serializer.serialize(command);
        zab.send(bb, context);
    }

    private void checkSnapshotInterval(final boolean afterStart) {
        if ((broadcasting
            && ((modifyCount >= modifySnapshotInterval
                || deliverCount >= deliverSnapshotInterval
                || transferredSize >= sizeSnapshotInterval)))
            || (afterStart
                && (modifyCount >> 2 >= modifySnapshotInterval
                    || deliverCount >> 2 >= deliverSnapshotInterval)))
        {
            try {
                modifyCount = 0;
                deliverCount = 0;
                transferredSize = 0;
                snapshot(false);
            } catch (ZabException e) {
                logger.log(
                    Level.SEVERE,
                    "Automatic snapshot taking error",
                    e);
            }
        }
    }

    @Override
    public ByteBuffer preprocess(
        final Zxid zxid,
        final ByteBuffer message)
    {
//        logger.fine("Preprocessing a message: " + message);
        try {
            if (preprocessInput == null) {
                preprocessInput = new ReusableByteBufferDataInput(message);
            } else {
                preprocessInput.reset(message);
            }
            Command command = Serializer.deserialize(preprocessInput);
            preprocessInput.clear();
//            logger.info("Preprocessing " + command);
            Command idempotent = command.preprocess(this);
            ByteBuffer bb = Serializer.serialize(idempotent);
            return bb;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    //CSOFF: ParameterNumber
    public void deliver(
        final Zxid zxid,
        final ByteBuffer stateUpdate,
        final String clientId,
        final Object ctx)
    {
        try {
            final int size = stateUpdate.remaining();
            if (deliverInput == null) {
                deliverInput = new ReusableByteBufferDataInput(stateUpdate);
            } else {
                deliverInput.reset(stateUpdate);
            }
            Command command =
                Serializer.deserialize(deliverInput);
            deliverInput.clear();
//            logger.info("Delivering " + command);
            command.execute(this);
            deliverCount++;
            transferredSize += size;
            checkSnapshotInterval(false);
            FutureCallback<Void> callback = (FutureCallback<Void>) ctx;
            if (callback == null) {
                // This request is sent from other instance.
                return;
            }
            callback.completed(null);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
    //CSON: ParameterNumber

    @Override
    public void flushed(
        final Zxid zxid,
        final ByteBuffer request,
        final Object ctx)
    {
    }

    @Override
    public void save(final FileOutputStream os) {
        logger.severe("Dumping snapshot");
        try {
            storage.save(os);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        logger.severe("Dumping snapshot done");
//        // No support for snapshot yet.
    }

    @Override
    public void restore(final FileInputStream is) {
        broadcasting = false;
        logger.severe("Loading snapshot");
        // No support for snapshot yet.
        try {
            storage.restore(is);
        } catch (Exception e) {
            logger.log(Level.SEVERE, "Snapshot restore error", e);
            throw new RuntimeException(e);
        }
        logger.severe("Loading snapshot done");
        deliverCount = 0;
    }

    @Override
    @SuppressWarnings("unchecked")
    public void snapshotDone(final String filePath, final Object ctx) {
        if (ctx != null) {
            FutureCallback<String> callback = (FutureCallback<String>) ctx;
            callback.completed(filePath);
        }
        logger.info("Snapshot write done to: " + filePath);
    }

    @Override
    public void removed(final String peerId, final Object ctx) {
    }

    @Override
    @SuppressWarnings("unchecked")
    public void recovering(final PendingRequests pendingRequests) {
        broadcasting = false;
        logger.info("Recovering...");
        // Returns error for all pending requests.
        IOException e = new IOException("Quorum is in recovering stage");
        for (Tuple tp : pendingRequests.pendingSends) {
            if (tp.ctx != null) {
                FutureCallback<Void> callback = (FutureCallback<Void>) tp.ctx;
                callback.failed(e);
            }
        }
        for (Tuple tp : pendingRequests.pendingFlushes) {
            if (tp.ctx != null) {
                FutureCallback<Void> callback = (FutureCallback<Void>) tp.ctx;
                callback.failed(e);
            }
        }
        for (Tuple tp : pendingRequests.pendingRemoves) {
            if (tp.ctx != null) {
                FutureCallback<Void> callback = (FutureCallback<Void>) tp.ctx;
                callback.failed(e);
            }
        }
        for (Object ctx : pendingRequests.pendingSnapshots) {
            if (ctx != null) {
                FutureCallback<Void> callback = (FutureCallback<Void>) ctx;
                callback.failed(e);
            }
        }
    }

    @Override
    public void leading(
        final Set<String> activeFollowers,
        final Set<String> clusterMembers)
    {
        logger.info("LEADING with active followers : ");
        for (String peer : activeFollowers) {
            logger.info(" -- " + peer);
        }
        logger.info("Cluster configuration change : " + clusterMembers.size());
        for (String peer : clusterMembers) {
            logger.info("  -- " + peer);
        }
        broadcasting = true;
        checkSnapshotInterval(true);
        isLeader = true;
    }

    @Override
    public void following(
        final String leader,
        final Set<String> clusterMembers)
    {
        logger.info("FOLLOWING " + leader);
        logger.info("Cluster configuration change  : " + clusterMembers.size());
        for (String peer : clusterMembers) {
            logger.info(" --  " + peer);
        }
        broadcasting = true;
        checkSnapshotInterval(true);
        isLeader = false;
    }

    public void cleanup() throws IOException, ZabException {
        add(
            new CleanupCommand(),
            new FutureCallback<Void>() {
                @Override
                public void completed(final Void v) {
                }

                @Override
                public void failed(final Exception e) {
                    logger.log(
                        Level.SEVERE,
                        "Cleanup error",
                        e);
                }

                @Override
                public void cancelled() {
                }
            });
    }

    public void dump() {
        storage.dump(server.logger().replacePrefix("Dump"));
    }

    public boolean isLeader() {
        return isLeader;
    }

    public boolean broadcasting() {
        return broadcasting;
    }

    public int size() {
        return storage.size();
    }

    public long weight() {
        return storage.weight();
    }

    public long maxWeight() {
        return storage.maxWeight();
    }

    private class Cleanuper extends Thread {
        Cleanuper() {
            super("Cleanuper");
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                try {
                    if (broadcasting && isLeader) {
                        cleanup();
                    }
                    Thread.sleep(CLEANUP_INTERVAL);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                } catch (Throwable t) {
                    logger.log(
                        Level.SEVERE,
                        "Cleanup failed",
                        t);
                }
            }
        }
    }
}
