package ru.yandex.jzabkv;

import com.github.zk1931.jzab.ZabException;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Collection;

import org.apache.http.Header;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.util.EntityUtils;

import ru.yandex.compress.GzipInputStream;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.jzabkv.command.PutCommand;
import ru.yandex.jzabkv.key.UTF8StringKey;
import ru.yandex.jzabkv.node.StringNode;
import ru.yandex.parser.string.DurationParser;

public class PutHandler implements ProxyRequestHandler {
    private final Database db;
    private final long defaultTTL;

    public PutHandler(final JZabKV server, final Database db) {
        this.db = db;
        this.defaultTTL = server.config().defaultTTL();
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        final UTF8StringKey key =
            new UTF8StringKey(session.params().getString("key"));
        final long ttl = session.params().get(
            "ttl",
            defaultTTL,
            DurationParser.POSITIVE_LONG);

        final StringNode node;
        final long evictionTimestamp = System.currentTimeMillis() + ttl;

        if (session.request() instanceof HttpEntityEnclosingRequest) {
            HttpEntityEnclosingRequest request =
                (HttpEntityEnclosingRequest) session.request();
            try {
                byte[] body = EntityUtils.toByteArray(request.getEntity());
                Header encoding = request.getFirstHeader("Content-Encoding");
                if (encoding != null && "gzip".equals(encoding.getValue())) {
                    body = gzipDecompress(body);
                }
                node = new StringNode(key, body, evictionTimestamp);
            } catch (IOException e) {
                throw new BadRequestException(e);
            }
        } else {
            final String value = session.params().getString("value");
            node = new StringNode(key, value, evictionTimestamp);
        }
        try {
            db.add(new PutCommand(node), new PutCallback(session));
        } catch (IOException | ZabException e) {
            throw new ServiceUnavailableException(e);
        }
    }

    private static byte[] gzipDecompress(byte[] data) throws IOException {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(data);
            ByteArrayOutputStream bos =
                new ByteArrayOutputStream(data.length << 1);
            GzipInputStream gzipIS = new GzipInputStream(bis))
        {
            byte[] buffer = new byte[4096];
            int len;
            while ((len = gzipIS.read(buffer)) != -1) {
                bos.write(buffer, 0, len);
            }
            return bos.toByteArray();
        }
    }

    private static final class PutCallback
        extends AbstractProxySessionCallback<Void>
    {
        PutCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Void v) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
