package ru.yandex.jzabkv;

import java.io.IOException;
import java.io.OutputStream;
//import java.io.UnsupportedEncodingException;

import java.nio.charset.StandardCharsets;

import java.util.Arrays;

import ru.yandex.jzabkv.node.Node;

public class ReusableDataOutputStream extends OutputStream {
    private static final int INT_SIZE = 4;

    protected byte[] buf;
    protected int count;

    public ReusableDataOutputStream() {
        this(INT_SIZE << 2);
    }

    public ReusableDataOutputStream(final int size) {
        if (size < 0) {
            throw new IllegalArgumentException(
                "Negative initial size: " + size);
        }
        buf = new byte[size];
    }

    private void ensureCapacity(final int minCapacity) {
        // overflow-conscious code
        if (minCapacity - buf.length > 0) {
            grow(minCapacity);
        }
    }

    private void grow(final int minCapacity) {
        // overflow-conscious code
        int oldCapacity = buf.length;
        int newCapacity = oldCapacity << 1;
        if (newCapacity - minCapacity < 0) {
            newCapacity = minCapacity;
        }
        if (newCapacity < 0) {
            if (minCapacity < 0) { // overflow
                throw new OutOfMemoryError();
            }
            newCapacity = Integer.MAX_VALUE;
        }
        buf = Arrays.copyOf(buf, newCapacity);
    }

    public void write(final int b) {
        ensureCapacity(count + 1);
        buf[count] = (byte) b;
        count += 1;
    }

    public void write(
        final byte[] b,
        final int off,
        final int len)
    {
        if ((off < 0) || (off > b.length) || (len < 0)
            || ((off + len) - b.length > 0))
        {
            throw new IndexOutOfBoundsException();
        }
        ensureCapacity(count + len);
        System.arraycopy(b, off, buf, count, len);
        count += len;
    }

    public void writeTo(final OutputStream out) throws IOException {
        out.write(buf, 0, count);
    }

    //CSOFF: MagicNumber
    public void writeInt(final int i) {
        ensureCapacity(count + INT_SIZE);
        buf[count++] = (byte) (i >> 24);
        buf[count++] = (byte) (i >> 16);
        buf[count++] = (byte) (i >> 8);
        buf[count++] = (byte) i;
    }
    //CSON: MagicNumber

    //CSOFF: MagicNumber
    public void writeLong(final long v) {
        writeInt((int) (v >> 32));
        writeInt((int) v);
    }
    //CSON: MagicNumber

    public void writeFloat(final float f) {
        writeInt(Float.floatToIntBits(f));
    }

    public void writeString(final String s) throws IOException {
        byte[] bytes = s.getBytes(StandardCharsets.UTF_8);
        writeInt(bytes.length);
        write(bytes);
    }

    public void writeNodeType(final Node node) {
        int ord = node.type().ordinal();
        writeInt(ord);
    }

    public void reset() {
        count = 0;
    }

    public byte toByteArray()[] {
        return Arrays.copyOf(buf, count);
    }

    public int size() {
        return count;
    }
/*
    public String toString() {
        return new String(buf, 0, count);
    }

    public String toString(final String charsetName)
        throws UnsupportedEncodingException
    {
        return new String(buf, 0, count, charsetName);
    }
*/
    public void close() throws IOException {
    }

    public byte[] buf() {
        return buf;
    }
}
