package ru.yandex.jzabkv;

import java.io.IOException;
import java.nio.ByteBuffer;

import ru.yandex.jzabkv.command.Command;
import ru.yandex.jzabkv.command.CommandType;
import ru.yandex.jzabkv.key.Key;
import ru.yandex.jzabkv.key.KeyType;
import ru.yandex.jzabkv.node.Node;
import ru.yandex.jzabkv.node.NodeType;

public final class Serializer {
    private static final ThreadLocal<ReusableDataOutputStream> RDOS =
        new ThreadLocal<ReusableDataOutputStream>() {
            @Override
            public ReusableDataOutputStream initialValue() {
                return new ReusableDataOutputStream();
            }
        };
    private static final CommandType[] COMMAND_TYPE_VALUES =
            CommandType.values();
    private static final KeyType[] KEY_TYPE_VALUES =
            KeyType.values();
    private static final NodeType[] NODE_TYPE_VALUES =
            NodeType.values();

    private Serializer() {
    }

    public static ByteBuffer serialize(final Command command)
        throws IOException
    {
        ReusableDataOutputStream rdos = RDOS.get();
        rdos.reset();
        rdos.writeInt(command.type().ordinal());
        command.serialize(rdos);
        return ByteBuffer.wrap(rdos.toByteArray());
    }

    public static void serialize(
        final Command command,
        final ReusableDataOutputStream rdos)
        throws IOException
    {
        rdos.writeInt(command.type().ordinal());
        command.serialize(rdos);
    }

    public static Command deserialize(final ReusableDataInput rdi)
        throws IOException
    {
        int typeIdx = rdi.getInt();
        CommandType type = COMMAND_TYPE_VALUES[typeIdx];
        return type.create(rdi);
    }

    public static Key deserializeKey(final ReusableDataInput rdi)
        throws IOException
    {
        int keyIdx = rdi.getInt();
        KeyType type = KEY_TYPE_VALUES[keyIdx];
        return type.create(rdi);
    }

    public static Node deserializeNode(
        final Key key,
        final ReusableDataInput rdi)
        throws IOException
    {
        int nodeIdx = rdi.getInt();
        NodeType type = NODE_TYPE_VALUES[nodeIdx];
        return type.create(key, rdi);
    }
}
