package ru.yandex.jzabkv.node;

import java.io.IOException;

import java.nio.charset.StandardCharsets;

import ru.yandex.jzabkv.key.Key;
import ru.yandex.jzabkv.ReusableDataInput;
import ru.yandex.jzabkv.ReusableDataOutputStream;

public class StringNode extends AbstractNode {
    private static final int OVERHEAD = 50;
    private byte[] value;

    public StringNode(
        final Key key,
        final byte[] value,
        final long evictionTimestamp)
    {
        super(key, evictionTimestamp);
        this.value = value;
    }

    public StringNode(
        final Key key,
        final String value,
        final long evictionTimestamp)
    {
        super(key, evictionTimestamp);
        if (value != null) {
            this.value = value.getBytes(StandardCharsets.UTF_8);
        } else {
            this.value = null;
        }
    }

    public StringNode(
        final Key key,
        final ReusableDataInput rdi)
        throws IOException
    {
        super(key, rdi);
        int len = rdi.getInt();
        if (len == 0) {
            value = null;
        } else {
            value = new byte[len];
            rdi.get(value);
        }
    }

    public String stringValue() {
        if (value != null) {
            return new String(value, StandardCharsets.UTF_8);
        } else {
            return null;
        }
    }

    @Override
    public byte[] bytes() {
        return value;
    }

    @Override
    public void writeData(final ReusableDataOutputStream stream)
        throws IOException
    {
        if (value == null) {
            stream.writeInt(0);
        } else {
            stream.writeInt(value.length);
            stream.write(value);
        }
    }

    @Override
    public NodeType type() {
        return NodeType.STRING;
    }

    @Override
    public String toString() {
        return "StringNode: value="
            + new String(value, StandardCharsets.UTF_8);
    }

    @Override
    public int weight() {
        if (value != null) {
            return OVERHEAD + value.length;
        } else {
            return OVERHEAD;
        }
    }
}
