package ru.yandex.jzabkv.storage;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;

import org.jctools.maps.NonBlockingHashMap;

import ru.yandex.jzabkv.ReusableDataInput;
import ru.yandex.jzabkv.ReusableDataOutputStream;
import ru.yandex.jzabkv.ReusableInputStreamDataInput;
import ru.yandex.jzabkv.Serializer;
import ru.yandex.jzabkv.key.Key;
import ru.yandex.jzabkv.node.Node;

public class NonBlockingHashMapStorage implements Storage {
    private static final int MAX_EVICT_COUNT = 100;
    private static final int WEIGHT_OVERHEAD = 128;

    private final NonBlockingHashMap<Key, Node> storage =
        new NonBlockingHashMap<>();
    private final ConcurrentSkipListMap<Long, Node> evictionMap =
        new ConcurrentSkipListMap<>();
    private final AtomicLong currentWeight = new AtomicLong(0);
    private final long maxWeight;
    private final Logger logger;

    public NonBlockingHashMapStorage(
        final long maxWeight,
        final Logger logger)
    {
        this.maxWeight = maxWeight;
        this.logger = logger;
    }

    @Override
    public void close() {
    }

    @Override
    public int size() {
        return storage.size();
    }

    @Override
    public long weight() {
        return currentWeight.get();
    }

    @Override
    public long maxWeight() {
        return maxWeight;
    }

    @Override
    public void clear() {
        storage.clear();
        evictionMap.clear();
        currentWeight.set(0);
    }

    @Override
    public Node get(final Key key) {
        return storage.get(key);
    }

    private int weight(final Node node) {
        return node.weight() + WEIGHT_OVERHEAD + node.key().weight();
    }

    @Override
    public void put(final Node node) {
//        logger.info("Putting: " + node.key());
        Node old = storage.put(node.key(), node);
        int weight = weight(node);
        if (old != null) {
//            logger.info("Replaced: " + old.key());
            weight -= weight(old);
            removeFromEvictionMap(old);
        }
        currentWeight.addAndGet(weight);
        Long ts = node.evictionTimestamp();
        old = evictionMap.putIfAbsent(ts, node);
        if (old != null) {
            while (old.next() != null) {
                old = old.next();
            }
            old.next(node);
        }
    }

    @Override
    public Collection<Key> tryPut(final Node node) {
        List<Key> toEvict = Collections.emptyList();

        Node old = storage.get(node.key());
        int weight = weight(node);
        if (old != null) {
            weight -= weight(old);
        }
        long overWeight = (currentWeight.get() + weight) - maxWeight;

        long currentTime = System.currentTimeMillis();
        Map.Entry<Long, Node> e = evictionMap.firstEntry();
        if (e != null && e.getKey() <= currentTime
            || overWeight > 0)
        {
            Iterator<Map.Entry<Long, Node>> iter =
                evictionMap.entrySet().iterator();
            toEvict = new ArrayList<>();
            evict:
            while (iter.hasNext()) {
                e = iter.next();
                for (
                    Node n = e.getValue();
                    n != null;
                    n = n.next())
                {
                    long evictTs = n.evictionTimestamp();
                    if ((evictTs <= currentTime || overWeight > 0)
                        && toEvict.size() < MAX_EVICT_COUNT)
                    {
                        toEvict.add(n.key());
                        overWeight -= weight(n);
                    } else {
                        break evict;
                    }
                }
            }
        }
        return toEvict;
    }

    private void removeFromEvictionMap(final Node node) {
//        logger.info("removeFromEvictionMap removing: " + node.key());
        Long ts = node.evictionTimestamp();
        Node head = evictionMap.get(ts);
        Node prev = null;
        if (head == null) {
            logger.severe("removeFromEvictionMap failed for node: "
                + node.key());
        }
        boolean deleted = false;
        for (Node n = head; n != null; prev = n, n = n.next()) {
            if (n == node) {
                //remove node
                if (n == head) {
                    if (head.next() == null) {
//                        logger.info("removeFromEvictionMap removed from map: " + node.key());
                        evictionMap.remove(ts);
                    } else {
//                        logger.info("removeFromEvictionMap replaced head: " + node.key());
                        evictionMap.put(ts, head.next());
                    }
                } else {
//                    logger.info("removeFromEvictionMap removed from list: " + node.key());
                    prev.next(n.next());
                }
                deleted = true;
                break;
            }
        }
        if (!deleted) {
            logger.severe("removeFromEvictionMap: node not deleted: "
                + node.key());
        } else {
//            logger.info("removeFromEvictionMap removed: " + node.key());
        }
    }

    @Override
    public void delete(final Key key) {
        Node node = storage.remove(key);
//        logger.info("Removing: " + key);
        if (node != null) {
//            logger.info("Removed: " + node.key());
            removeFromEvictionMap(node);
            currentWeight.addAndGet(-weight(node));
        }
    }

    @Override
    public void save(final FileOutputStream os) throws IOException {
        DataOutputStream dos =
            new DataOutputStream(
                new BufferedOutputStream(os));
        ReusableDataOutputStream ros = new ReusableDataOutputStream();
        for (Map.Entry<Key, Node> entry: storage.entrySet()) {
            Key key = entry.getKey();
            Node node = entry.getValue();
            dos.writeBoolean(true);
            ros.reset();

            key.write(ros);
            node.write(ros);

            dos.write(ros.buf(), 0, ros.size());
        }
        dos.writeBoolean(false);
        dos.flush();
    }

    @Override
    public void restore(final FileInputStream is) throws IOException {
        ReusableDataInput rdi =
            new ReusableInputStreamDataInput(
                new BufferedInputStream(is));
        storage.clear();
        while (rdi.getBoolean()) {
            Key key = Serializer.deserializeKey(rdi);
            Node node = Serializer.deserializeNode(key, rdi);
            put(node);
        }
    }

    @Override
    public void dump(final Logger logger) {
        for (Map.Entry<Long, Node> e: evictionMap.entrySet()) {
            Node n = e.getValue();
            if (!storage.containsKey(n.key())) {
                logger.severe("Orphaned eviction node: " + n.key()
                    + ", ts: " + e.getKey()
                    + ", next: " + (n.next() == null ? "null" : n.next().key()));
            }
        }
    }
}
