package ru.yandex.search.so;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import com.github.jelmerk.knn.SearchResult;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncResponseProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.logger.PrefixedLogger;

public class DeletePointHandler implements HttpAsyncRequestHandler<JsonObject>
{
    private final Knn knn;

    public DeletePointHandler(final Knn knn) {
        this.knn = knn;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
            final HttpRequest request,
            final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncTypesafeDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE);
    }

/*
{
    "ns":["hot_spam"],
    "point":{
        "coordinate":$COORDS,
        "value":{
            "spam":true,
            "stid":"$STID",
            "uid":"$PUID",
            "smtp_id":"$QUEUE_ID"
        }
    }
}
*/
    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
//        ProxySession session = new BasicProxySession(proxy, exchange, context);
        try {
            JsonMap request = payload.asMap();
            JsonMap point = request.getMap("point");
            MinHashMessage msg = new MinHashMessage(point);
            double distance = request.getDouble("distance", 0.0);
            boolean sync = request.getBoolean("sync", false);
            boolean byId = request.getBoolean("by-id", false);
            int k = request.getInt("k", 10);
            PrefixedLogger logger =
                (PrefixedLogger) context.getAttribute(HttpProxy.LOGGER);
            List<MinHashMessage> toDelete = new ArrayList<>();
            boolean noMoreElements = false;
            if (byId) {
                toDelete.add(msg);
            } else {
                while (toDelete.size() < k && !noMoreElements) {
                    List<SearchResult<MinHashMessage, Float>> neighbors =
                        knn.neighbors(msg.vector(), k);
                    for (SearchResult<MinHashMessage, Float> n: neighbors) {
                        if (n.distance() < distance) {
                            toDelete.add(n.item());
                        } else {
                            noMoreElements = true;
                            break;
                        }
                    }
                }
            }
            if (toDelete.size() == 0) {
                logger.severe("No points found to delete");
                BasicAsyncResponseProducerGenerator respGenerator =
                    new BasicAsyncResponseProducerGenerator(
                        HttpStatus.SC_OK,
                        "Ok");
                exchange.submitResponse(respGenerator.get());
                return;
            }
            logger.severe("Found " + toDelete.size() + " points to delete");
            MinHashMessage last = toDelete.remove(toDelete.size() - 1);
            for (MinHashMessage m: toDelete) {
                knn.deletePoint(m, null);
            }
            if (sync) {
                knn.deletePoint(
                    last,
                    new SyncCallback(knn, exchange, context, logger));
            } else {
                knn.deletePoint(last, null);
                logger.severe("Deleted points, index size: " + knn.indexSize());
                BasicAsyncResponseProducerGenerator respGenerator =
                    new BasicAsyncResponseProducerGenerator(
                        HttpStatus.SC_OK,
                        "Ok");
                exchange.submitResponse(respGenerator.get());
            }
        } catch (JsonException|IOException e) {
            throw new BadRequestException(e);
        }
    }

    private static class SyncCallback implements FutureCallback<Integer> {
        private final Knn knn;
        private final HttpAsyncExchange exchange;
        private final HttpContext context;
        private final Logger logger;

        SyncCallback(
            final Knn knn,
            final HttpAsyncExchange exchange,
            final HttpContext context,
            final Logger logger)
        {
            this.knn = knn;
            this.exchange = exchange;
            this.context = context;
            this.logger = logger;
        }

        @Override
        public void completed(final Integer result) {
            logger.severe("Delete point <sync>, index size: " + result);
            BasicAsyncResponseProducerGenerator respGenerator =
                new BasicAsyncResponseProducerGenerator(
                        HttpStatus.SC_OK,
                        "Ok");
            exchange.submitResponse(respGenerator.get());
        }

        @Override
        public void cancelled() {
        }

        @Override
        public void failed(final Exception e) {
            exchange.submitResponse(knn.handleException(e, context));
        }
    }
}

