package ru.yandex.search.so;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.jniwrapper.ImmutableJniWrapperConfig;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.ValuesStorage;
import ru.yandex.util.string.UnhexStrings;

public class I2TCalc {
    private final Options opts;
    private final JniWrapper dssm;

    private I2TCalc(final Options opts) throws Exception {
        this.opts = opts;
        ImmutableJniWrapperConfig dssmConfig =
            new ImmutableJniWrapperConfig(
                new JniWrapperConfigBuilder()
                    .libraryName(
                        opts.getString("library", "libdisk-dssm-jniwrapper.so"))
                    .ctorName(
                        opts.getString("ctor", "DssmApplierCreateInstance"))
                    .dtorName(
                        opts.getString("dtor", "DssmApplierDestroyInstance"))
                    .mainName(
                        opts.getString("main", "DssmApplierGetFeatures"))
                    .config(
                        opts.getString("config", "i2t_ver7_txt_prod.cfg")));
        dssm =
            JniWrapper.create(
                dssmConfig,
                new ThreadFactoryConfig("Jni-")
                    .daemon(true));
    }

    public static void main(final String[] args) throws Exception {
        Options opts = Options.parseArgs(args, "path");
        I2TCalc i2t = new I2TCalc(opts);
        i2t.calc();
    }

    private void calc() throws Exception {
        Iterator<String> paths = opts.getAllOrNull("path");
        if (paths == null) {
            paths = Collections.singletonList(":stdin:").iterator();
            return;
        }
        while (paths.hasNext()) {
            String file = paths.next();
            try (BufferedReader reader =
                new BufferedReader(
                    new InputStreamReader(
                        createInputStream(file),
                        StandardCharsets.UTF_8)))
            {
                while ((file = reader.readLine()) != null) {
                    byte[] vector = vector(file);
                    System.out.println(
                        file + " v " + vector.length + " "
                        + vectorString(vector));
                }
            }
        }
    }

    private byte[] vector(final String file) throws Exception {
        String content =
            new String(Files.readAllBytes(Paths.get(file)), "UTF-8");
        String i2t = dssm.apply(content, null).process(new byte[1], 0, 1);
        byte[] vector = UnhexStrings.unhex(i2t);
        return vector;
    }

    private static String vectorString(final byte[] data) {
        StringBuilder sb = new StringBuilder();
        String sep = "";
        for (int i = 0; i < data.length; i++) {
            int v = data[i];
            sb.append(sep);
            sb.append(v);
            sep = " ";
        }
        return new String(sb);
    }

    private static InputStream createInputStream(final String file)
        throws IOException
    {
        if (file.equals(":stdin:")) {
            return System.in;
        } else {
            return new FileInputStream(file);
        }
    }

    private static class Options extends IniConfig
        implements ValuesStorage<ConfigException>
    {
        private static final long serialVersionUID = -7862295268420722926L;
        private final Map<String, String> aliases = new HashMap<>();
        private final Map<String, String> desc = new HashMap<>();

        Options() throws ConfigException, IOException {
            super(new StringReader(""));
        }

        public void opt(
            final String name,
            final String alias,
            final String desc)
        {
            aliases.put(name, alias);
            this.desc.put(name, desc);
        }

        public static Options parseArgs(
            final String[] args,
            final String defaultOpt)
            throws ConfigException, IOException
        {
            boolean searchKey = true;
            String key = null;
            Options opts = new Options();
            for (int i = 0; i < args.length; i++) {
                String token = args[i];
                if (searchKey) {
                    if (token.charAt(0) == '-') {
                        key = token.substring(1);
                        searchKey = false;
                    } else {
                        opts.put(defaultOpt, token);
                    }
                } else {
                    opts.put(key, token);
                    searchKey = true;
                    key = null;
                }
            }
            return opts;
        }

        @Override
        public String getOrNull(final String name) {
            String value = super.getOrNull(name);
            if (value == null) {
                String alias = aliases.get(name);
                if (alias != null) {
                    value = super.getOrNull(alias);
                }
            }
            return value;
        }

/*
        @Override
        public void put(final String key, final int value) {
            this.put(key, String.valueOf(value));
        }

        public void put(final String key, final long value) {
            this.put(key, String.valueOf(value));
        }

        public void put(final String key, final double value) {
            this.put(key, String.valueOf(value));
        }

        @Override
        public ConfigException parameterNotSetException(final String name) {
            return new ConfigException("No field with name " + name);
        }

        @Override
        public ConfigException parseFailedException(
            final String name,
            final String value,
            final Throwable cause)
        {
            return new ConfigException(
                "Parse failed for field " + name + " with value " + value,
                cause);
        }

        @Override
        public String getOrNull(final String name) {
            String value = get(name);
            if (value == null) {
                String alias = aliases.get(name);
                if (alias != null) {
                    value = get(alias);
                }
            }
            return value;
        }

        @Override
        public String getLastOrNull(final String name) {
            return getOrNull(name);
        }

        @Override
        public Iterator<String> getAllOrNull(final String name) {
            String value = getOrNull(name);
            if (value == null) {
                return null;
            }

            Iterator<String> result = null;
            try {
                result = new CollectionParser<>(
                    NonEmptyValidator.TRIMMED,
                    ArrayList::new).apply(value).iterator();
            } catch (Exception e) {
                e.printStackTrace();
            }

            return result;
        }
*/
    }

}
