package ru.yandex.search.so;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.atomic.LongAdder;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.config.DnsConfigBuilder;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.server.BaseServerConfigBuilder;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.ValuesStorage;
import ru.yandex.util.timesource.TimeSource;

public class KnnUpload {
    private final Options opts;
    private final float minDist;
    private final HttpHost host;
    private final AsyncClient client;
    private final boolean sync;

    private KnnUpload(final Options opts) throws Exception {
        this.opts = opts;
        this.client = createClient(opts);
        this.minDist = (float) opts.getDouble("min-distance");
        //this.host = new HttpHost(opts.getString("host"));
        String hostPort = opts.getString("host");
        int sep = hostPort.indexOf(':');
        int port = 80;
        if (sep != -1) {
            port = Integer.parseInt(hostPort.substring(sep + 1));
            this.host = new HttpHost(hostPort.substring(0, sep), port);
        } else {
            this.host = new HttpHost(hostPort);
        }
        sync = opts.getBoolean("sync", false);
    }

    public static void main(final String[] args) throws Exception {
        Options opts = Options.parseArgs(args, "file");
        opts.opt("threads", "t", "Reactor thread count");
        opts.opt("connections", "c", "Max connection count");
        opts.opt("connect-timeout", "ct", "Connect timeout");
        opts.opt("socket-timeout", "st", "Socket timeout");
        opts.opt("host", "h", "host");
        opts.opt("count", "n", "count");
        opts.opt("min-distance", "md", "minimal distance to add");
        opts.opt("sync", "s", "sync add");

        KnnUpload ku = new KnnUpload(opts);
        ku.upload();
    }

    private void upload() throws Exception {
        int conns = opts.getInt("connections", 10);
        {
            Iterator<String> files = opts.getAllOrNull("file");
            if (files == null) {
                System.err.println("No files specified");
                return;
            }
            ArrayBlockingQueue<Boolean> inFlight = new ArrayBlockingQueue<>(conns * 2);
            Callback callback = new Callback(inFlight);

            JsonParser parser;
            BasicGenericConsumer<JsonObject, JsonException> consumer;
            consumer = new BasicGenericConsumer<>();
            parser =
                new JsonParser(new StackContentHandler(
                    new TypesafeValueContentHandler(consumer)));
            while (files.hasNext()) {
                String file = files.next();
                try (BufferedReader reader =
                    new BufferedReader(
                        new InputStreamReader(
                            new FileInputStream(file),
                            StandardCharsets.UTF_8)))
                {
                    for (
                        String line = reader.readLine();
                        line != null;
                        line = reader.readLine())
                    {
                        line = line.trim();
                        if (line.length() == 0) {
                            continue;
                        }
                        try {
                            parser.parse(line);
                            JsonObject o = consumer.get();
                            JsonMap map = o.asMap();
                            String uid = map.getString("rcpt_uid", null);
                            if (uid == null) {
                                uid = map.getString("uid", null);
                                if (uid == null) {
                                    continue;
                                }
                            }
                            String queueId =
                                map.getString("x-yandex-queueid", null);
                            if (queueId == null) {
                                queueId = map.getString(
                                    "x_yandex_queueid",
                                    null);
                                if (queueId == null) {
                                    continue;
                                }
                            }
                            String minhash = map.getString("minhash", null);
                            if (minhash == null || minhash.length() == 0) {
                                continue;
                            }
//                            System.out.println("min: " + minhash);
                            inFlight.put(Boolean.valueOf(true));
                            checkAndAdd(
                                uid,
                                queueId,
                                minhash,
                                callback);
//                            System.out.println("\rProcessed: " + (++i));
                        } catch (JsonException e) {
                        }
                    }
                }
            }
        }
    }

    private void checkAndAdd(
        final String uid,
        final String queueId,
        final String minhash,
        final FutureCallback<Void> callback)
    {
        String json = "{\"n\":1,\"coordinate\":[" + minhash + "]}";
        BasicAsyncRequestProducerGenerator request =
            new BasicAsyncRequestProducerGenerator(
                "/knn/neighbors",
                json);
        client.execute(
            host,
            request,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            new NeighborsCallback(
                uid,
                queueId,
                minhash,
                callback));
    }

    private static AsyncClient createClient(final Options opts)
        throws Exception
    {
        ImmutableHttpTargetConfig clientConfig;
        ImmutableBaseServerConfig serverConfig;
        ImmutableDnsConfig dnsConfig;
        dnsConfig = new DnsConfigBuilder(opts).build();
        int connections = opts.getInt("connections", 100);
        int threads = opts.getInt("threads", 1);
        int connectTimeout = opts.getIntegerDuration("connect-timeout", 100);
        int socketTimeout = opts.getIntegerDuration("socket-timeout", 30000);
        serverConfig =
            new BaseServerConfigBuilder()
                    .port(0)
                    .name("Client")
                    .connections(connections)
                    .workers(threads)
                    .build();

        ImmutableHttpTargetConfig targetConfig =
            new HttpTargetConfigBuilder()
                .connections(connections)
                .connectTimeout(connectTimeout)
                .timeout(socketTimeout)
                .poolTimeout(connectTimeout)
                .build();

        clientConfig =
            new HttpTargetConfigBuilder(targetConfig)
                .briefHeaders(true)
                .build();
        SharedConnectingIOReactor reactor =
            new SharedConnectingIOReactor(serverConfig, dnsConfig);
        AsyncClient client = new AsyncClient(reactor, clientConfig);
        reactor.start();
        client.start();
        return client;
    }

    private static class Options extends IniConfig
        implements ValuesStorage<ConfigException>
    {
        private static final long serialVersionUID = -7862295268420722926L;
        private final Map<String, String> aliases = new HashMap<>();
        private final Map<String, String> desc = new HashMap<>();

        Options() throws ConfigException, IOException {
            super(new StringReader(""));
        }

        public void opt(
            final String name,
            final String alias,
            final String desc)
        {
            aliases.put(name, alias);
            this.desc.put(name, desc);
        }

        public static Options parseArgs(
            final String[] args,
            final String defaultOpt)
            throws ConfigException, IOException
        {
            boolean searchKey = true;
            String key = null;
            Options opts = new Options();
            for (int i = 0; i < args.length; i++) {
                String token = args[i];
                if (searchKey) {
                    if (token.charAt(0) == '-') {
                        key = token.substring(1);
                        searchKey = false;
                    } else {
                        opts.put(defaultOpt, token);
                    }
                } else {
                    opts.put(key, token);
                    searchKey = true;
                    key = null;
                }
            }
            return opts;
        }

        @Override
        public String getOrNull(final String name) {
            String value = super.getOrNull(name);
            if (value == null) {
                String alias = aliases.get(name);
                if (alias != null) {
                    value = super.getOrNull(alias);
                }
            }
            return value;
        }

/*
        @Override
        public void put(final String key, final int value) {
            this.put(key, String.valueOf(value));
        }

        public void put(final String key, final long value) {
            this.put(key, String.valueOf(value));
        }

        public void put(final String key, final double value) {
            this.put(key, String.valueOf(value));
        }

        @Override
        public ConfigException parameterNotSetException(final String name) {
            return new ConfigException("No field with name " + name);
        }

        @Override
        public ConfigException parseFailedException(
            final String name,
            final String value,
            final Throwable cause)
        {
            return new ConfigException(
                "Parse failed for field " + name + " with value " + value,
                cause);
        }

        @Override
        public String getOrNull(final String name) {
            String value = get(name);
            if (value == null) {
                String alias = aliases.get(name);
                if (alias != null) {
                    value = get(alias);
                }
            }
            return value;
        }

        @Override
        public String getLastOrNull(final String name) {
            return getOrNull(name);
        }

        @Override
        public Iterator<String> getAllOrNull(final String name) {
            String value = getOrNull(name);
            if (value == null) {
                return null;
            }

            Iterator<String> result = null;
            try {
                result = new CollectionParser<>(
                    NonEmptyValidator.TRIMMED,
                    ArrayList::new).apply(value).iterator();
            } catch (Exception e) {
                e.printStackTrace();
            }

            return result;
        }
*/
    }

    private class NeighborsCallback
        implements FutureCallback<JsonObject>
    {
        private final String uid;
        private final String queueId;
        private final String minhash;
        private final FutureCallback<Void> callback;

        NeighborsCallback(
            final String uid,
            final String queueId,
            final String minhash,
            final FutureCallback<Void> callback)
        {
            this.uid = uid;
            this.queueId = queueId;
            this.minhash = minhash;
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject o) {
            try {
                JsonMap map = o.asMap();
                JsonList found = map.getListOrNull("hot_spam");
                if (found != null && found.size() > 0) {
                    JsonMap first = found.get(0).asMap();
                    Double dist = first.getDouble("distance", null);
                    if (dist != null && dist <= minDist) {
                        //System.out.println("Skipping: " + dist);
                        callback.cancelled();
                        return;
                    }
                }
                String json = "{"
                    + "\"sync\":" + sync + ","
                    + "\"ns\":[\"hot_spam\"],\"point\":{"
                    + "\"coordinate\":[" + minhash + "],"
                    + "\"value\":{"
                    + "\"spam\":true,"
                    + "\"uid\":\"" + uid + "\","
                    + "\"smtp_id\":\"" + queueId + "\"}}}";
                BasicAsyncRequestProducerGenerator request =
                new BasicAsyncRequestProducerGenerator(
                    "/knn/add_point",
                    json);
                client.execute(
                    host,
                    request,
                    EmptyAsyncConsumerFactory.OK,
                    callback);
            } catch (Exception e) {
                callback.failed(e);
            }
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }

    private static class Callback implements FutureCallback<Void> {
        private final ArrayBlockingQueue<Boolean> inFlight;
        private final LongAdder oks = new LongAdder();
        private final LongAdder skips = new LongAdder();
        private final LongAdder total = new LongAdder();
        private long prevStats = TimeSource.INSTANCE.currentTimeMillis();
        private long prevTotal = 0;

        Callback(final ArrayBlockingQueue<Boolean> inFlight) {
            this.inFlight = inFlight;
        }

        @Override
        public void completed(final Void v) {
            inFlight.poll();
            oks.add(1);
            total.add(1);
//            System.out.println("\rOks: " + oks.incrementAndGet());
            stats();
        }

        private void stats() {
            long time = TimeSource.INSTANCE.currentTimeMillis();
            if (time - prevStats > 500) {
                long newTotal = total.sum();
                prevStats = time;
                System.out.println("\rOks: " + oks.sum()
                    + ", Skips: " + skips.sum()
                    + ", rps: "
                    + ((newTotal - prevTotal) * 2));
                prevTotal = newTotal;
            }
        }

        @Override
        public void failed(final Exception e) {
            inFlight.poll();
            System.out.println("Err: " + e.getMessage());
        }

        @Override
        public void cancelled() {
            skips.add(1);
            total.add(1);
            inFlight.poll();
            stats();
        }
    }
}
