package ru.yandex.search.so;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Reader;
import java.util.Arrays;

import com.github.jelmerk.knn.Item;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;

public class MinHashMessage implements Item<String, int[]> {
    private static final ThreadLocal<SimpleJsonParser> JSON_PARSER =
        ThreadLocal.<SimpleJsonParser>withInitial(
            () -> new SimpleJsonParser());
    private static final ThreadLocal<SimpleJsonWriter> JSON_WRITER =
        ThreadLocal.<SimpleJsonWriter>withInitial(
            () -> new SimpleJsonWriter());
    private static final long serialVersionUID = 42L;
    private final String id;
    private final int[] vector;
    private final byte[] payload;

    private MinHashMessage() {
        id = null;
        vector = null;
        payload = null;
    }

    public MinHashMessage(final JsonMap json)
        throws IOException, JsonException
    {
        JsonList coords = json.getList("coordinate");
        JsonMap value = json.getMap("value");
        id = value.getString("smtp_id");
        payload = jsonToByteArray(value);
        vector = createVector(coords);
    }

    public MinHashMessage(final String id, final int[] vector) {
        this.id = id;
        this.vector = vector;
        this.payload = null;
    }

    public static int[] createVector(final JsonList list) throws JsonException {
        int[] vector = new int[list.size()];
        int i = 0;
        for (JsonObject o: list) {
            vector[i++] = (int) o.asLong();
        }
        return vector;
    }

    public static byte[] jsonToByteArray(final JsonObject o)
        throws IOException, JsonException
    {
        return JSON_WRITER.get().toByteArray(o);
    }

    @Override
    public String id() {
        return id;
    }

    @Override
    public int[] vector() {
        return vector;
    }

    public byte[] payload() {
        return payload;
    }

    @Override
    public String toString() {
        return "Msg{" + id + "}: "
            + Arrays.toString(vector);
    }

    public static JsonObject parseJson(final String str)
        throws IOException, JsonException
    {
        return JSON_PARSER.get().parse(str);
    }

    public static JsonObject parseJson(final Reader reader)
        throws IOException, JsonException
    {
        return JSON_PARSER.get().parse(reader);
    }

    private static final class SimpleJsonParser {
        private final JsonParser parser;
        private final BasicGenericConsumer<JsonObject, JsonException> consumer;

        SimpleJsonParser() {
            consumer = new BasicGenericConsumer<>();
            parser =
                new JsonParser(new StackContentHandler(
                    new TypesafeValueContentHandler(consumer)));
        }

        public JsonObject parse(final String json)
            throws IOException, JsonException
        {
            parser.parse(json);
            return consumer.get();
        }

        public JsonObject parse(final Reader reader)
            throws IOException, JsonException
        {
            parser.parse(reader);
            return consumer.get();
        }
    }

    private static final class SimpleJsonWriter {
        private final Utf8JsonWriter writer;
        private final ByteArrayOutputStream out;

        SimpleJsonWriter() {
            out = new ByteArrayOutputStream();
            writer = JsonType.NORMAL.create(out);
        }

        public byte[] toByteArray(final JsonObject o) throws IOException {
            out.reset();
            o.writeValue(writer);
            writer.flush();
            writer.reset();
            return out.toByteArray();
        }
    }
}
