package ru.yandex.search.so;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.logging.Level;

import com.github.jelmerk.knn.SearchResult;
import org.apache.http.Header;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;

public class NeighborsHandler implements HttpAsyncRequestHandler<JsonObject>
{
    private final Knn knn;
    private final int defaultK;

    public NeighborsHandler(final Knn knn) {
        this.knn = knn;
        defaultK = knn.config().defaultSearchK();
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
            final HttpRequest request,
            final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        Header encoding = request.getFirstHeader("Content-Encoding");
        if (encoding != null && "gzip".equals(encoding.getValue())) {
            return new GzipEntityAsyncConsumer();
        }
        return new JsonAsyncTypesafeDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE);
    }

/*
{
    "ns":["hot_spam"],
    "point":{
        "coordinate":$COORDS,
        "value":{
            "spam":true,
            "stid":"$STID",
            "uid":"$PUID",
            "smtp_id":"$QUEUE_ID"
        }
    }
}
*/
    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(knn, exchange, context);
        try {
            JsonMap request = payload.asMap();
            JsonList coordinate = request.getList("coordinate");
            int k = request.getInt("n", defaultK);
            int[] vector = MinHashMessage.createVector(coordinate);
            PrefixedLogger logger =
                (PrefixedLogger) context.getAttribute(HttpProxy.LOGGER);
            if (logger.isLoggable(Level.FINEST)) {
                logger.finest("request: " + JsonType.NORMAL.toString(payload));
            }
            JsonType jsonType =
                JsonTypeExtractor.NORMAL.extract(session.params());
            knn.neighbors(
                vector,
                k,
                new ResultsCallback(
                    session,
                    jsonType,
                    vector,
                    k));
        } catch (JsonException e) {
            throw new BadRequestException(e);
        }
    }

    private class ResultsCallback
        extends AbstractProxySessionCallback<
            List<SearchResult<MinHashMessage, Float>>>
        implements Runnable
    {
        private final JsonType jsonType;
        private final int[] vector;
        private final int k;

        ResultsCallback(
            final ProxySession session,
            final JsonType jsonType,
            final int[] vector,
            final int k)
        {
            super(session);
            this.jsonType = jsonType;
            this.vector = vector;
            this.k = k;
        }

        @Override
        public void run() {
            List<SearchResult<MinHashMessage, Float>> neighbors =
                knn.neighbors(vector, k);
            completed(neighbors);
        }

        @Override
        public void completed(
            final List<SearchResult<MinHashMessage, Float>> result)
        {
            try {
                StringBuilderWriter sbw = new StringBuilderWriter();
                JsonWriter writer = jsonType.create(sbw);
                writer.startObject();
                writer.key("hot_spam");
                writer.startArray();
                for (SearchResult<MinHashMessage, Float> res: result) {
                    MinHashMessage msg = res.item();
                    writer.startObject();
                    writer.key("distance");
                    writer.value(res.distance());
                    writer.key("value");
                    if (msg.payload() != null) {
//                          msg.payload().asMap().getMap("value").writeValue(writer);
                        writer.jsonValue(
                            new String(msg.payload(), StandardCharsets.UTF_8));
                    } else {
                        writer.value((String) null);
                    }
                    writer.endObject();
                }
                writer.endArray();
                writer.endObject();
                writer.close();
                session.response(HttpStatus.SC_OK, sbw.toString());
            } catch (IOException e) {
                failed(e);
            }
        }

        @Override
        public void cancelled() {
        }
    }
}

