package ru.yandex.search.passport.korobochka;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.collection.IntPair;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.passport.korobochka.common.AbstractLogRecord;

public class KorobochkaSender {
    private static final ContentType KOROBOCHKA_CONTENT_TYPE =
        ContentType.TEXT_PLAIN.withCharset(StandardCharsets.UTF_8);

    private final UpstreamStater korobochkaStater;
    private final AsyncClient client;
    private final String authString;
    private final HttpHost korobochkaHost;
    private final String name;
    private final String uri;

    public KorobochkaSender(
        final Korobochka korobochka,
        final ImmutableURIConfig config,
        final String login,
        final String password,
        final String name)
        throws ConfigException, IOException
    {
        this.name = name;
        this.uri = config.request();
//        this.korobochka = korobochka;
        korobochkaHost = config.host();

        client = korobochka.client("Korobochka-" + name, config);

        String authString = login
            + ':' + password;
        Base64Encoder encoder = new Base64Encoder();
        encoder.process(authString.getBytes(StandardCharsets.UTF_8));
        this.authString = "Basic " + encoder.toString();

        korobochkaStater =
            new UpstreamStater(
                korobochka.config().metricsTimeFrame(),
                "korobochka-" + name);
        korobochka.registerStater(korobochkaStater);
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public final void sendRecords(
        final ProxySession session,
        final List<? extends AbstractLogRecord> records,
        final FutureCallback<Void> callback,
        final TimeFrameQueue<Long> recordsSkipped)
        throws IOException
    {
        sendRecords(
            session,
            records,
            new Callback(session, callback, recordsSkipped));
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private final void sendRecords(
        final ProxySession session,
        final List<? extends AbstractLogRecord> records,
        final Callback callback)
        throws IOException
    {
        callback.records(records);
        AsyncClient client = this.client.adjust(session.context());
//        final String uri = "/tables/yandex-user/records";
        final byte[] postData;
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.reset();
        try (OutputStreamWriter osw =
            new OutputStreamWriter(baos, StandardCharsets.UTF_8))
        {
//            sbw.append(
            for (AbstractLogRecord record: records) {
                session.logger().info(
                    name + " Sending record: " + record.toString());
                osw.write(record.toString());
                osw.write('\n');
            }
            osw.flush();
            postData = baos.toByteArray();
        }
        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                KOROBOCHKA_CONTENT_TYPE);
        post.addHeader("Authorization", authString);
        client.execute(
            korobochkaHost,
            post,
            new StatusCheckAsyncResponseConsumerFactory<IntPair<JsonObject>>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                new StatusCodeAsyncConsumerFactory<JsonObject>(
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE)),
            session.listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                callback,
                korobochkaStater));
    }

    private class Callback extends AbstractFilterFutureCallback<
        IntPair<JsonObject>,
        Void>
    {
        private final ProxySession session;
        private List<? extends AbstractLogRecord> records;
        private final TimeFrameQueue<Long> recordsSkipped;

        Callback(
            final ProxySession session,
            final FutureCallback<? super Void> callback,
            final TimeFrameQueue<Long> recordsSkipped)
        {
            super(callback);
            this.session = session;
            this.recordsSkipped = recordsSkipped;
        }

        public void records(final List<? extends AbstractLogRecord> records) {
            this.records = records;
        }

        private void parseResponse(final JsonObject o) {
            try {
                JsonMap json = o.asMap();
                int total = json.getInt("total");
                boolean interrupted = json.getBoolean("interrupted", false);
                if (total < records.size()) {
                    session.logger().severe("Korobochka total < record count: "
                        + total + " < " + records.size());
                    if (interrupted) {
                        session.logger().severe("Korobochka handling internal error: "
                            + "interrupted on " + total + " record. "
                            + "Resending remaining"
                            + " records");
                        Callback nextCallback = clone();
                        List<? extends AbstractLogRecord> nextRecords =
                            records.subList(total, records.size());
                        nextCallback.records(nextRecords);
                        try {
                            sendRecords(
                                session,
                                nextRecords,
                                nextCallback);
                            return;
                        } catch (IOException e) {
                            session.logger().severe("Korobochka resend error");
                            failed(e);
                        }
                    } else {
                        session.logger().severe("Korobochka handling internal error: "
                            + "can't handle " + total + " record. "
                            + "Resending remaining"
                            + " records");
                        Callback nextCallback = clone();
                        recordsSkipped.accept(1L);
                        List<? extends AbstractLogRecord> nextRecords =
                            records.subList(total + 1, records.size());
                        nextCallback.records(nextRecords);
                        try {
                            sendRecords(
                                session,
                                nextRecords,
                                nextCallback);
                            return;
                        } catch (IOException e) {
                            session.logger().severe("Korobochka resend error");
                            failed(e);
                        }
                    }
                } else {
                    callback.completed(null);
                }
            } catch (JsonException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Json parse exception",
                    e);
                failed(e);
            }
        }

        @Override
        public void completed(final IntPair<JsonObject> v) {
            if (v.first() == 0) {
                session.logger().warning("Skipped all records");
                callback.completed(null);
                return;
            }
            session.logger().warning("Korobochka response: "
                + "code=" + v.first()
                + ", body=" + JsonType.NORMAL.toString(v.second()));
            if (v.first() < HttpStatus.SC_MULTIPLE_CHOICES) {
                parseResponse(v.second());
            } else if (v.first() < HttpStatus.SC_GONE) {
                session.logger().warning("Skipping records");
                callback.completed(null);
                return;
            } else {
                String msg = "Unhandled korobochka response code: "
                    + v.first();
                session.logger().severe(msg);
                failed(new BadGatewayException(msg));
            }
        }

        @Override
        public Callback clone() {
            return new Callback(session, callback, recordsSkipped);
        }
    }


    public void close() throws IOException {
        client.close();
    }
}
