package ru.yandex.search.passport.korobochka;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.protocol.HTTP;
import org.apache.http.util.EntityUtils;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.MimeTokenStream;
//import org.apache.http.concurrent.FutureCallback;

import ru.yandex.compress.GzipInputStream;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.search.passport.korobochka.avtoru.AvtoRuHandler;
import ru.yandex.search.passport.korobochka.passport.AccountModificationHandler;
import ru.yandex.search.passport.korobochka.passport.HistoryDB2Handler;

public class LogbrokeItHandler implements ProxyRequestHandler
{
    private static final int TEMP_BUFFER_SIZE = 4096;

    private final Korobochka korobochka;
    private final AccountModificationHandler acountModificationHandler;
    private final HistoryDB2Handler historyDB2Handler;
    private final AvtoRuHandler avtoRuHandler;
    private final UnhandledHandler unhandledHandler;

    public LogbrokeItHandler(
        final Korobochka korobochka)
    {
        this.korobochka = korobochka;
        acountModificationHandler =
            new AccountModificationHandler(korobochka);
        historyDB2Handler = new HistoryDB2Handler(korobochka);
        avtoRuHandler = new AvtoRuHandler(korobochka);
        unhandledHandler = new UnhandledHandler();
    }

    public Korobochka korobochka() {
        return korobochka;
    }

    private TopicHandler selectHandler(final ProxySession session)
        throws HttpException
    {
        final String topic = session.params().getString("topic");
        if (topic.endsWith("passport-account-modification-log")
            || topic.endsWith("passport-account-modification-log-fast-testing")
            || topic.endsWith("passport_reginfo"))
        {
            return acountModificationHandler;
        } else if (topic.contains("auto_ru")) {
            return avtoRuHandler;
        } else if (topic.contains("historydb2")) {
            return historyDB2Handler;
        } else {
            return unhandledHandler;
        }
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        HttpEntityEnclosingRequest request =
            (HttpEntityEnclosingRequest) session.request();
        byte[] body = EntityUtils.toByteArray(request.getEntity());
        Header encoding = request.getFirstHeader("Content-Encoding");
        if (encoding != null && "gzip".equals(encoding.getValue())) {
            body = gzipDecompress(body);
        }

        final String contentTypeHeader =
            session.headers().getString(HTTP.CONTENT_TYPE, null);
        ContentType contentType = null;
        if (contentTypeHeader != null) {
            try {
                contentType = ContentType.parse(contentTypeHeader);
            } catch (Exception e) {
                throw new BadRequestException("Bad content-type header");
            }
        }
        final boolean multiPart;
        if (contentType != null
            && contentType.getMimeType().equals("multipart/mixed"))
        {
            multiPart = true;
        } else {
            multiPart = false;
        }
        List<byte[]> bodies;
        if (multiPart) {
            bodies = parseMultipart(session, body);
            session.logger().info("Multipart bodies: " + bodies.size());
        } else {
            bodies = Collections.singletonList(body);
        }

        TopicHandler handler = selectHandler(session);
        session.logger().info("Selected handler: " + handler.getClass().getName());
        handler.handle(session, bodies);

//        executor.execute(
//            new TskvBufferParser(
//                session,
//                bodies,
//                new Callback(session, korobochka)));
    }

    private static byte[] gzipDecompress(byte[] data) throws IOException {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(data);
            ByteArrayOutputStream bos =
                new ByteArrayOutputStream(data.length << 1);
            GzipInputStream gzipIS = new GzipInputStream(bis))
        {
            byte[] buffer = new byte[4096];
            int len;
            while ((len = gzipIS.read(buffer)) != -1) {
                bos.write(buffer, 0, len);
            }
            return bos.toByteArray();
        }
    }

    private List<byte[]> parseMultipart(
        final ProxySession session,
        final byte[] data)
        throws HttpException
    {
        List<byte[]> parts = new ArrayList<>();
        MimeTokenStream stream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        byte[] tempBuffer = new byte[TEMP_BUFFER_SIZE];
        try {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) session.request()).getEntity();
            stream.parseHeadless(
                new ByteArrayInputStream(data),
                entity.getContentType().getValue());
            EntityState state = stream.getState();
            while (state != EntityState.T_END_OF_STREAM) {
                switch (state) {
                    case T_BODY:
                        int pos = 0;
                        try (InputStream is = stream.getDecodedInputStream())
                                //stream.getBodyDescriptor().getCharset()))
                        {
                            int read =
                                is.read(
                                    tempBuffer,
                                    pos,
                                    tempBuffer.length - pos);
                            while (read != -1) {
                                pos += read;
                                if (pos == tempBuffer.length) {
                                    tempBuffer = Arrays.copyOf(
                                        tempBuffer,
                                        tempBuffer.length << 1);
                                }
                                read = is.read(
                                    tempBuffer,
                                    pos,
                                    tempBuffer.length - pos);
                            }
                        }
                        byte[] body = Arrays.copyOf(tempBuffer, pos);
                        parts.add(body);
                        break;
                    default:
                        break;
                }
                state = stream.next();
            }
        } catch (Throwable t) {
            throw new NotImplementedException(t);
        }
        return parts;
    }

    private static class UnhandledHandler implements TopicHandler {
        @Override
        public void handle(final ProxySession session, final List<byte[]> bodies) {
            try {
                session.logger().severe("Unhandled topic name: "
                    + session.params().getString("topic"));
                session.response(HttpStatus.SC_NOT_IMPLEMENTED);
            } catch (BadRequestException e) {
                failed(session, e);
            }
        }
    }
}
