package ru.yandex.search.passport.korobochka.common;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import org.apache.http.HttpStatus;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.passport.korobochka.ImmutableKorobochkaConfig;
import ru.yandex.search.passport.korobochka.Korobochka;
import ru.yandex.search.passport.korobochka.KorobochkaSender;
import ru.yandex.search.passport.korobochka.TopicHandler;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.tskv.TskvParser;

@SuppressWarnings("ThreadLocalUsage")
public abstract class AbstractLogHandler<T extends AbstractLogRecord>
    implements TopicHandler, LogHandler<T>
{
    private final ThreadLocal<ParserWithHandler<T>> tskvParser =
        new ThreadLocal<>();
    protected final Korobochka korobochka;
    private final ThreadPoolExecutor executor;
    private final TimeFrameQueue<Long> recordsProcessed;
    private final TimeFrameQueue<Long> recordsSent;
    private final TimeFrameQueue<Long> recordsSkipped;
    private final KorobochkaSender korobochkaSender;
    private final String statsPrefix;

    protected AbstractLogHandler(
        final Korobochka korobochka,
        final KorobochkaSender korobochkaSender,
        final String statsPrefix)
    {
        this.korobochka = korobochka;
        this.korobochkaSender = korobochkaSender;
        this.statsPrefix = statsPrefix;
        ImmutableKorobochkaConfig config = korobochka.config();

        recordsProcessed = new TimeFrameQueue<>(config.metricsTimeFrame());
        recordsSent = new TimeFrameQueue<>(config.metricsTimeFrame());
        recordsSkipped = new TimeFrameQueue<>(config.metricsTimeFrame());

        executor =
            new ThreadPoolExecutor(
                config.parseThreads(),
                config.parseThreads(),
                1,
                TimeUnit.HOURS,
                new ArrayBlockingQueue<>(config.parseQueue()));
        korobochka.registerStater(
            new PassiveStaterAdapter<>(
                recordsProcessed,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        statsPrefix + "_records_processed_ammm",
                        CountAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        statsPrefix + "_records_processed_axxx",
                        CountAggregatorFactory.INSTANCE))));
        korobochka.registerStater(
            new PassiveStaterAdapter<>(
                recordsSent,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        statsPrefix + "_records_sent_ammm",
                        CountAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        statsPrefix + "_records_sent_axxx",
                        CountAggregatorFactory.INSTANCE))));
        korobochka.registerStater(
            new PassiveStaterAdapter<>(
                recordsSkipped,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        statsPrefix + "_records_skipped_ammm",
                        CountAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        statsPrefix + "_records_skipped_axxx",
                        CountAggregatorFactory.INSTANCE))));
    }

    public Korobochka korobochka() {
        return korobochka;
    }

    @Override
    public void processRecords(final long count) {
        recordsProcessed.accept(count);
    }

    public void sendRecords(final long count) {
        recordsSent.accept(count);
    }

    public void skipRecords(final long count) {
        recordsSkipped.accept(count);
    }

    @Override
    public boolean nonTskv() {
        return false;
    }

    @Override
    public T parseRecord(final Logger logger, final String line) {
        throw new UnsupportedOperationException("parseRecord is not "
            + "defined while handler declared as non tskv: "
            + this.getClass().getName());
    }

    public ParserWithHandler<T> parser() {
        ParserWithHandler<T> ph = tskvParser.get();
        if (ph == null) {
            ph = new ParserWithHandler<T>(this) {
                @Override
                public T createRecord() {
                    return AbstractLogHandler.this.createRecord();
                }
            };
            tskvParser.set(ph);
        }
        return ph;
    }

    @Override
    public void handle(final ProxySession session, final List<byte[]> bodies) {
        executor.execute(
            new TskvBufferParser(
                session,
                bodies,
                new ResponseCallback(session)));
    }

    private void processRecords(
        final ProxySession session,
        final List<T> records,
        final ResponseCallback callback)
        throws IOException
    {
        MultiFutureCallback<T> multiCallback =
            new MultiFutureCallback<>(
                new ProcessedRecordsCallback(
                    session,
                    callback));
        for (T record: records) {
            processRecord(record, session,  multiCallback.newCallback());
        }
        multiCallback.done();
    }

    private void postProcessRecords(
        final ProxySession session,
        final List<T> records,
        final ResponseCallback callback)
        throws IOException
    {
        Iterator<T> iter = records.iterator();
        while (iter.hasNext()) {
            T record = iter.next();
            if (record.skipRecord()) {
                iter.remove();
            }
        }
        sendRecords(records.size());
        session.logger().fine("Sending " + records.size()
            + " records");
        korobochkaSender.sendRecords(session, records, callback, recordsSkipped);
    }

    private class TskvBufferParser implements Runnable {
        private final ProxySession session;
        private final List<byte[]> bodies;
        private final ResponseCallback callback;

        TskvBufferParser(
            final ProxySession session,
            final List<byte[]> bodies,
            final ResponseCallback callback)
        {
            this.session = session;
            this.bodies = bodies;
            this.callback = callback;
        }

        @Override
        public void run() {
            ParserWithHandler<T> parser = parser();
            parser.reset(session.logger());
            try {
                if (nonTskv()) {
                    for (byte[] body: bodies) {
                        try (BufferedReader reader =
                            new BufferedReader(
                                new InputStreamReader(
                                    new ByteArrayInputStream(body),
                                    StandardCharsets.UTF_8)))
                        {
                            for (
                                String line = reader.readLine();
                                line != null;
                                line = reader.readLine())
                            {
                                T record = parseRecord(session.logger(), line);
                                if (record != null) {
                                    parser.handler.onRecord(record);
                                }
                            }
                        }
                    }
                } else {
                    for (byte[] body: bodies) {
                        String str = new String(body, StandardCharsets.UTF_8);
                        parser.parse(new StringReader(str));
                    }
                }
                List<T> records = parser.filteredRecords();
                if (records.size() > 0) {
                    processRecords(session, records, callback);
                } else {
                    session.logger().severe("Filtered out all records");
                    callback.completed(null);
                }
            } catch (Exception e) {
                callback.failed(e);
            }
        }
    }

    private class ProcessedRecordsCallback
        extends AbstractProxySessionCallback<List<T>>
    {
        private final ResponseCallback callback;

        ProcessedRecordsCallback(
            final ProxySession session,
            final ResponseCallback callback)
        {
            super(session);
            this.callback = callback;
        }

        @Override
        public void completed(final List<T> records) {
            try {
                postProcessRecords(session, records, callback);
            } catch (IOException e) {
                failed(e);
            }
        }
    }

    private static abstract class ParserWithHandler<T extends AbstractLogRecord>
        extends TskvParser<T>
    {
        private final AbstractLogHandler<T> logHandler;
        private final TskvRecordHandler<T> handler;

        ParserWithHandler(
            final TskvRecordHandler<T> handler,
            final AbstractLogHandler<T> logHandler)
        {
            super(handler);
            this.handler = handler;
            this.logHandler = logHandler;
        }

        ParserWithHandler(final AbstractLogHandler<T> parent) {
            this(new TskvRecordHandler<T>(parent), parent);
        }

        public void reset(final PrefixedLogger logger) {
            handler.clear();
            handler.logger(logger.addPrefix(logHandler.statsPrefix));
            super.reset();
        }

        public List<T> filteredRecords() {
            return handler.filteredRecords();
        }

//        @Override
//        public T createRecord() {
//            return logHandler.createRecord();
//        }
    }

    private static class ResponseCallback
        extends AbstractProxySessionCallback<Void>
    {
        ResponseCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Void v) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
