package ru.yandex.search.passport.korobochka.passport;

import java.nio.CharBuffer;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.search.passport.korobochka.Korobochka;
import ru.yandex.search.passport.korobochka.common.AbstractLogHandler;

public class HistoryDB2Handler extends AbstractLogHandler<HistoryDB2Record> {
    private static final char DELIMITER = ' ';
    private static final char QUOTE = '`';
    private static final int HISTORYDB2_FIELD_COUNT =
        EventField.values().length;

    private enum EventField {
        VERSION(0, "version"),
        RFC_TIME(1, "rfc_time"),
        HOST_ID(2, "host_id"),
        CLIENT_NAME(3, "client_name"),
        UID(4, "uid"),
        EVENT_NAME(5, "event_name"),
        EVENT_VALUE(6, "event_value"),
        USER_IP(7, "user_ip"),
        PROXY_IP(8, "proxy_ip"),
        YANDEX_UID(9, "yandex_uid"),
        ADMIN_LOGIN(10, "admin_login"),
        COMMENT(11, "comment");

        private EventField(final int column, final String name) {
            this.column = column;
            this.name = name;
        }

        private final int column;
        private final String name;
    }

    public HistoryDB2Handler(
        final Korobochka korobochka)
    {
        super(korobochka, korobochka.passportSender(), "historydb2");
    }

    @Override
    public HistoryDB2Record createRecord() {
        return new HistoryDB2Record();
    }

    @Override
    public boolean nonTskv() {
        return true;
    }

    private void unquote(
        final StringBuilder sb,
        final CharBuffer cb,
        final ArrayList<String> fields)
        throws ParseException
    {
        cb.get();
        sb.setLength(0);
        while (cb.hasRemaining()) {
            char c = cb.get();
            if (c == QUOTE) {
                if (cb.hasRemaining() && cb.get(cb.position() + 1) == QUOTE) {
                    sb.append(QUOTE);
                    cb.get();
                } else {
                    fields.add(new String(sb));
                    return;
                }
            } else {
                sb.append(c);
            }
        }
        throw new ParseException("Bad quoted field: " + sb, cb.position());
    }

    private void readField(
        final StringBuilder sb,
        final CharBuffer cb,
        final ArrayList<String> fields)
    {
        sb.setLength(0);
        while (cb.hasRemaining()) {
            char c = cb.get();
            if (c == DELIMITER) {
                fields.add(new String(sb));
                return;
            } else {
                sb.append(c);
            }
        }
        if (sb.length() > 0) {
            fields.add(new String(sb));
        }
    }

    private ArrayList<String> fields(final String line) throws ParseException {
        CharBuffer cb = CharBuffer.wrap(line);
        ArrayList<String> fields = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        while (cb.hasRemaining()) {
            char first = cb.get(cb.position());
            if (first == QUOTE) {
                unquote(sb, cb, fields);
                if (cb.hasRemaining()) {
                    cb.get();
                }
            } else {
                readField(sb, cb, fields);
            }
        }
        return fields;
    }

    @Override
    public HistoryDB2Record parseRecord(
        final Logger logger,
        final String line)
    {
        try {
            ArrayList<String> fields = fields(line);
            if (fields.size() != HISTORYDB2_FIELD_COUNT) {
                logger.severe("HISTORYDB.count: " + fields.size()
                    + ": " + fields.toString());
                return null;
            }

            HistoryDB2Record record = createRecord();
            for (EventField field: EventField.values()) {
                record.put(field.name, fields.get(field.column));
            }
            return record;
        } catch (ParseException e) {
            logger.log(
                Level.SEVERE,
                "csv line parse error",
                e);
            return null;
        }
    }

    @Override
    public void filterRecord(final HistoryDB2Record record) {
        record.skipRecord(true);
    }

    @Override
    public void processRecord(
        final HistoryDB2Record record,
        final ProxySession session,
        final FutureCallback<HistoryDB2Record> callback)
    {
        record.skipRecord(true);
        callback.completed(record);
    }
}
