package ru.yandex.search.mail.kamaji;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.http.config.FilterSearchConfig;
import ru.yandex.http.config.FilterSearchConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.search.mail.kamaji.usertype.UserTypeConfig;
import ru.yandex.search.mail.kamaji.usertype.UserTypeConfigBuilder;

public abstract class AbstractKamajiConfigBuilder
    <T extends AbstractKamajiConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements KamajiConfig
{
    private static final String KAMAJI_SECTION = "kamaji";

    private HttpHostConfigBuilder backendConfig;
    private HttpHostConfigBuilder searchConfig;
    private HttpHostConfigBuilder slowIndexerConfig;
    private HttpHostConfigBuilder blackboxConfig;
    private HttpHostConfigBuilder tikaiteConfig;
    private Tvm2ClientConfigBuilder tvm2ClientConfig;
    private String tikaiteTvmClientId;
    private String unistorageTvmClientId;
    private String blackboxTvmClientId;
    private String filterSearchTvmClientId;
    private FilterSearchConfigBuilder filterSearchConfig;
    private UserTypeConfigBuilder userTypeConfig;
    private Set<String> preserveFields;
    private Pattern mdbs;
    private boolean subscriptionsIndexEnabled;
    private long tikaiteMemoryLimit;
    private int tikaiteMemoryQueueCapacity;

    protected AbstractKamajiConfigBuilder(final KamajiConfig config) {
        super(config);
        backendConfig(new HttpHostConfigBuilder(config.backendConfig()));
        searchConfig(new HttpHostConfigBuilder(config.searchConfig()));
        slowIndexerConfig(
            new HttpHostConfigBuilder(config.slowIndexerConfig()));
        blackboxConfig(new HttpHostConfigBuilder(config.backendConfig()));
        tikaiteConfig(new HttpHostConfigBuilder(config.tikaiteConfig()));
        tvm2ClientConfig(
            new Tvm2ClientConfigBuilder(config.tvm2ClientConfig()));
        tikaiteTvmClientId(config.tikaiteTvmClientId());
        unistorageTvmClientId(config.unistorageTvmClientId());
        blackboxTvmClientId(config.blackboxTvmClientId());
        filterSearchTvmClientId(config.filterSearchTvmClientId());
        filterSearchConfig(
            new FilterSearchConfigBuilder(config.filterSearchConfig()));
        userTypeConfig(new UserTypeConfigBuilder(config.userTypeConfig()));
        preserveFields(config.preserveFields());
        mdbs(config.mdbs());
        subscriptionsIndexEnabled(config.subscriptionsIndexEnabled());
        tikaiteMemoryLimit(config.tikaiteMemoryLimit());
        tikaiteMemoryQueueCapacity(config.tikaiteMemoryQueueCapacity());
    }

    protected AbstractKamajiConfigBuilder(
        final IniConfig config,
        final KamajiConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        backendConfig = new HttpHostConfigBuilder(
            config.section("backend"),
            defaults.backendConfig());
        searchConfig = new HttpHostConfigBuilder(
            config.section("backend_search"),
            defaults.searchConfig());
        slowIndexerConfig = new HttpHostConfigBuilder(
            config.section("slow-indexer"),
            defaults.slowIndexerConfig());

        IniConfig blackboxSection = config.section("blackbox");
        blackboxConfig = new HttpHostConfigBuilder(
            blackboxSection,
            defaults.blackboxConfig());
        IniConfig tikaiteSection = config.section("tikaite");
        tikaiteConfig = new HttpHostConfigBuilder(
            tikaiteSection,
            defaults.tikaiteConfig());
        tvm2ClientConfig = new Tvm2ClientConfigBuilder(
            config.section("tvm2"),
            defaults.tvm2ClientConfig());
        tikaiteTvmClientId = config.get(
            "tikaite.tvm-client-id",
            defaults.tikaiteTvmClientId(),
            NonEmptyValidator.INSTANCE);
        unistorageTvmClientId =
            config.get(
                "tikaite.unistorage-client-id",
                defaults.unistorageTvmClientId(),
                NonEmptyValidator.INSTANCE);
        blackboxTvmClientId = config.get(
            "blackbox.tvm-client-id",
            defaults.blackboxTvmClientId(),
            NonEmptyValidator.INSTANCE);
        filterSearchTvmClientId = config.get(
            "filter-search.tvm-client-id",
            defaults.filterSearchTvmClientId(),
            NonEmptyValidator.INSTANCE);
        filterSearchConfig = new FilterSearchConfigBuilder(
            config.section("filter-search"),
            defaults.filterSearchConfig());
        preserveFields = config.section(KAMAJI_SECTION).get(
            "preserve-fields",
            defaults.preserveFields(),
            new CollectionParser<>(String::trim, HashSet::new));
        mdbs = config.section(KAMAJI_SECTION).get(
            "mdbs",
            defaults.mdbs(),
            Pattern::compile);

        subscriptionsIndexEnabled =
            config.section(KAMAJI_SECTION).getBoolean(
                "subscriptions-enabled",
                defaults.subscriptionsIndexEnabled());

        IniConfig userTypeSection = config.sectionOrNull("user-type");
        if (userTypeSection != null) {
            userTypeConfig = new UserTypeConfigBuilder(
                userTypeSection,
                defaults.userTypeConfig());
        } else {
            userTypeConfig =
                new UserTypeConfigBuilder(defaults.userTypeConfig());
        }

        tikaiteMemoryLimit =
            tikaiteSection.get(
                "memory-limit",
                defaults.tikaiteMemoryLimit(),
                PositiveLongValidator.INSTANCE);
        tikaiteMemoryQueueCapacity =
            tikaiteSection.get(
                "memory-limit-queue-size",
                defaults.tikaiteMemoryQueueCapacity(),
                PositiveIntegerValidator.INSTANCE);
    }

    @Override
    public HttpHostConfigBuilder backendConfig() {
        return backendConfig;
    }

    public T backendConfig(final HttpHostConfig backendConfig) {
        this.backendConfig = new HttpHostConfigBuilder(backendConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder searchConfig() {
        return searchConfig;
    }

    public T searchConfig(final HttpHostConfig config) {
        this.searchConfig = new HttpHostConfigBuilder(config);
        return self();
    }

    @Override
    public HttpHostConfigBuilder slowIndexerConfig() {
        return slowIndexerConfig;
    }

    public T slowIndexerConfig(final HttpHostConfig slowIndexerConfig) {
        this.slowIndexerConfig = new HttpHostConfigBuilder(slowIndexerConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder blackboxConfig() {
        return blackboxConfig;
    }

    public T blackboxConfig(final HttpHostConfig blackboxConfig) {
        this.blackboxConfig = new HttpHostConfigBuilder(blackboxConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder tikaiteConfig() {
        return tikaiteConfig;
    }

    public T tikaiteConfig(final HttpHostConfig tikaiteConfig) {
        this.tikaiteConfig = new HttpHostConfigBuilder(tikaiteConfig);
        return self();
    }

    @Override
    public Tvm2ClientConfigBuilder tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public T tvm2ClientConfig(final Tvm2ClientConfig tvm2ClientConfig) {
        this.tvm2ClientConfig = new Tvm2ClientConfigBuilder(tvm2ClientConfig);
        return self();
    }

    @Override
    public String tikaiteTvmClientId() {
        return tikaiteTvmClientId;
    }

    public T tikaiteTvmClientId(final String tikaiteTvmClientId) {
        this.tikaiteTvmClientId = tikaiteTvmClientId;
        return self();
    }

    @Override
    public String unistorageTvmClientId() {
        return unistorageTvmClientId;
    }

    public T unistorageTvmClientId(final String unistorageTvmClientId) {
        this.unistorageTvmClientId = unistorageTvmClientId;
        return self();
    }

    @Override
    public String blackboxTvmClientId() {
        return blackboxTvmClientId;
    }

    public T blackboxTvmClientId(final String blackboxTvmClientId) {
        this.blackboxTvmClientId = blackboxTvmClientId;
        return self();
    }

    @Override
    public String filterSearchTvmClientId() {
        return filterSearchTvmClientId;
    }

    public T filterSearchTvmClientId(final String filterSearchTvmClientId) {
        this.filterSearchTvmClientId = filterSearchTvmClientId;
        return self();
    }

    @Override
    public FilterSearchConfigBuilder filterSearchConfig() {
        return filterSearchConfig;
    }

    public T filterSearchConfig(final FilterSearchConfig filterSearchConfig) {
        this.filterSearchConfig =
            new FilterSearchConfigBuilder(filterSearchConfig);
        return self();
    }

    @Override
    public Pattern mdbs() {
        return mdbs;
    }

    public T mdbs(final Pattern mdbs) {
        this.mdbs = mdbs;
        return self();
    }

    @Override
    public Set<String> preserveFields() {
        return preserveFields;
    }

    public T preserveFields(final Set<String> fields) {
        this.preserveFields = fields;
        return self();
    }

    @Override
    public UserTypeConfig userTypeConfig() {
        return userTypeConfig;
    }

    public T userTypeConfig(final UserTypeConfig config) {
        this.userTypeConfig = new UserTypeConfigBuilder(config);
        return self();
    }

    @Override
    public boolean subscriptionsIndexEnabled() {
        return subscriptionsIndexEnabled;
    }

    public T subscriptionsIndexEnabled(
        final boolean subscriptionsIndexEnabled)
    {
        this.subscriptionsIndexEnabled = subscriptionsIndexEnabled;
        return self();
    }

    @Override
    public long tikaiteMemoryLimit() {
        return tikaiteMemoryLimit;
    }

    public T tikaiteMemoryLimit(final long tikaiteMemoryLimit) {
        this.tikaiteMemoryLimit = tikaiteMemoryLimit;
        return self();
    }

    @Override
    public int tikaiteMemoryQueueCapacity() {
        return tikaiteMemoryQueueCapacity;
    }

    public T tikaiteMemoryQueueCapacity(final int tikaiteMemoryQueueCapacity) {
        this.tikaiteMemoryQueueCapacity = tikaiteMemoryQueueCapacity;
        return self();
    }
}

