package ru.yandex.search.mail.kamaji;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.http.HttpException;

import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.dbfields.PgFields;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.search.mail.kamaji.senders.SendersIndexerModuleFactory;
import ru.yandex.search.mail.kamaji.sherlock.SherlockTemplatedIdIndexerFactory;
import ru.yandex.search.mail.kamaji.subscriptions.IndexModule;
import ru.yandex.search.mail.kamaji.subscriptions.IndexModuleFactory;
import ru.yandex.search.mail.kamaji.subscriptions.NewSubscriptionsModuleFactory;
import ru.yandex.search.mail.kamaji.subscriptions.SlowIndexModule;
import ru.yandex.search.mail.kamaji.subscriptions.SlowIndexModuleFactory;
import ru.yandex.search.mail.kamaji.subscriptions.SubsIndexModuleFactory;

public class ChangeContext {
    private static final double MILLIS = 1000d;

    private static final List<IndexModuleFactory> DEFAULT_SUB_INDEXERS =
        Arrays.asList(SherlockTemplatedIdIndexerFactory.INSTANCE);

    private static final List<SlowIndexModuleFactory> SLOW_SUB_INDEXERS =
        Arrays.asList(SendersIndexerModuleFactory.INSTANCE);

    private static final String SENDERS = "senders";

    private final Kamaji kamaji;
    private final Map<?, ?> json;
    private final ProxySession session;
    private final double operationDate;

    private final long queueId;
    private final long deadline;
    private final boolean slow;
    private final boolean markAsRead;

    private final long prefix;
    private final ChangeType changeType;
    private final Long lcn;
    private final int offset;
    private final Map<String, Map<?, ?>> updates;
    private final Map<MailSearchDatabases, List<IndexModule>> subIndexers;
    // should request user actions for mail before proceeding indexation
    private final boolean fetchClicks;
    private final List<SlowIndexModule> slowSubIndexers;

    private final Collection<String> preserveFields;
    private final Collection<String> preserveSlowFields;

    public ChangeContext(
        final Kamaji kamaji,
        final ProxySession session,
        final Map<?, ?> json)
        throws HttpException, JsonException
    {
        this.kamaji = kamaji;
        this.session = session;
        this.json = json;
        operationDate =
            ValueUtils.asDouble(json.get(OracleFields.OPERATION_DATE));
        slow = session.params().getBoolean("slow", false);
        deadline = session.params().getLong("deadline", Long.MAX_VALUE);

        Long queueId = new HeadersParser(session.request())
            .getLong(YandexHeaders.ZOO_QUEUE_ID, null);
        if (queueId == null || queueId < 0) {
            this.queueId = session.params().getLong("zoo-queue-id", -1L);
        } else {
            this.queueId = queueId;
        }

        prefix = ValueUtils.asLong(json.get(PgFields.UID));
        try {
            changeType = ValueUtils.asEnum(
                ChangeType.class,
                json.get(PgFields.CHANGE_TYPE));
        } catch (JsonException e) {
            throw new ServiceUnavailableException(
                "Failed to parse change type",
                e);
        }

        offset = session.params().get(
            "offset",
            0,
            NonNegativeIntegerValidator.INSTANCE);

        preserveFields =
            new LinkedHashSet<>(kamaji.config().preserveFields());
        if (changeType == ChangeType.FIELDS_UPDATE) {
            lcn = null;

            updates = new HashMap<>();

            for (Object docObj: ValueUtils.asList(json.get(PgFields.CHANGED))) {
                Map<?, ?> doc = new HashMap<>(ValueUtils.asMap(docObj));
                String mid = ValueUtils.asString(doc.get(PgFields.MID));

                doc.remove(PgFields.MID);
                doc.remove(SENDERS);

                updates.put(mid, doc);
            }

            preserveFields.add(OracleFields.LCN);
        } else {
            lcn = ValueUtils.asLong(json.get(OracleFields.LCN));
            updates = Collections.emptyMap();
        }

        Map<?, ?> arguments =
            ValueUtils.asMapOrNull(json.get(PgFields.ARGUMENTS));
        markAsRead = arguments != null
            && ValueUtils.asBooleanOrDefault(arguments.get("seen"), false);

        subIndexers = new LinkedHashMap<>();
        List<IndexModule> defaultSubIndexers = new ArrayList<>();
        // todo passing not finalized object is bad
        // special hook subs module, todo avoid it
        IndexModule subsModule = SubsIndexModuleFactory.INSTANCE.create(this);
        if (subsModule != null) {
            defaultSubIndexers.add(subsModule);
            fetchClicks = changeType == ChangeType.MOVE
                || changeType == ChangeType.FIELDS_UPDATE;
        } else {
            fetchClicks = false;
        }

        for (IndexModuleFactory factory: DEFAULT_SUB_INDEXERS) {
            IndexModule module = factory.create(this);
            if (module != null) {
                defaultSubIndexers.add(module);
            }
        }

        subIndexers.put(MailSearchDatabases.DEFAULT, defaultSubIndexers);
        IndexModule newSubsModule = NewSubscriptionsModuleFactory.INSTANCE.create(this);
        if (newSubsModule != null) {
            subIndexers.put(
                MailSearchDatabases.SUBSCRIPTIONS,
                Collections.singletonList(newSubsModule));
        }

        for (IndexModule module: defaultSubIndexers) {
            preserveFields.addAll(module.preserveFields());
        }

        preserveSlowFields = new ArrayList<>();
        slowSubIndexers = new ArrayList<>();
        if (slow) {
            for (SlowIndexModuleFactory factory: SLOW_SUB_INDEXERS) {
                SlowIndexModule module = factory.create(this);
                if (module != null) {
                    slowSubIndexers.add(module);
                    preserveSlowFields.addAll(module.preserveFields());
                }
            }
        }
    }

    public Kamaji kamaji() {
        return kamaji;
    }

    public ProxySession session() {
        return session;
    }

    public Map<?, ?> json() {
        return json;
    }

    public double operationDate() {
        return operationDate;
    }

    public long zooQueueId() {
        return queueId;
    }

    public long operationDateMillis() {
        return (long) (operationDate * MILLIS);
    }

    public boolean slow() {
        return slow;
    }

    public boolean markAsRead() {
        return markAsRead;
    }

    public boolean corp() {
        return BlackboxUserinfo.corp(prefix());
    }

    public String humanReadableJson() {
        return JsonType.HUMAN_READABLE.toString(json);
    }

    public String userTag() {
        return prefixType().name().toLowerCase(Locale.ROOT) + '=' + prefix();
    }

    public long deadline() {
        return deadline;
    }

    public long prefix() {
        return prefix;
    }

    public BlackboxUserIdType prefixType() {
        return BlackboxUserIdType.UID;
    }

    public boolean reindex() {
        return changeType == ChangeType.REINDEX;
    }

    public ChangeType changeType() {
        return changeType;
    }

    public Long lcn() {
        return lcn;
    }

    public int offset() {
        return offset;
    }

    public Map<String, Map<?, ?>> updates() {
        return updates;
    }

    public Collection<String> preserveFields() {
        return preserveFields;
    }

    public Collection<String> preserveSlowFields() {
        return preserveSlowFields;
    }

    public Map<MailSearchDatabases, List<IndexModule>> dbSubIndexers() {
        return subIndexers;
    }

    public boolean fetchClicks() {
        return fetchClicks;
    }

    public List<SlowIndexModule> slowSubIndexers() {
        return slowSubIndexers;
    }

    public AsyncClient adjustClient(final AsyncClient client) {
        if (changeType == ChangeType.FIELDS_UPDATE) {
            return client.addHeader(
                YandexHeaders.X_INDEX_OPERATION_QUEUE,
                "clicks");
        } else {
            return client;
        }
    }

    public String url(final String mid, final String hid) {
        StringBuilder sb = new StringBuilder();
        sb.append(prefix);
        sb.append('_');
        sb.append(mid);
        sb.append('/');
        sb.append(hid);
        return new String(sb);
    }
}

