package ru.yandex.search.mail.kamaji;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.dbfields.PgFields;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock.LockAcquireStatus;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock.LockRequest;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock.LockerType;
import ru.yandex.util.string.StringUtils;

public enum DeletePgHandler implements ChangeHandler {
    INSTANCE;

    private static final int CHAR_PER_LONG = 19;
    private static final String DELETE_PREFIX = "/delete?prefix=";
    private static final String REQUEST = "&text=mid_p:(";
    private static final int CAPACITY_OVERHEAD =
        DELETE_PREFIX.length() + CHAR_PER_LONG + REQUEST.length();

    @Override
    public void handle(final ChangeContext context)
        throws HttpException, JsonUnexpectedTokenException
    {
        BasicIndexCallback callback = new BasicIndexCallback(context);
        int size = callback.changed().size();
        if (size == 0) {
            throw new BadRequestException(
                "Empty 'changed' list: " + context.json());
        }
        StringBuilder sb =
            new StringBuilder(CAPACITY_OVERHEAD + (CHAR_PER_LONG + 1) * size);
        sb.append(DELETE_PREFIX);
        sb.append(context.prefix());
        sb.append(REQUEST);

        List<LockCallback> locks = new ArrayList<>(size);

        DeleteIndexCallback indexCallback =
            new DeleteIndexCallback(context, locks);

        LocksCallback locksCallback =
            new LocksCallback(context, indexCallback);

        MultiFutureCallback<Object> multiCallback =
            new MultiFutureCallback<>(locksCallback);

        for (Object change : callback.changed()) {
            Long mid = ValueUtils.asLong(
                ValueUtils.asMap(change).get(PgFields.MID));
            sb.append(mid);
            sb.append('+');

            String key =
                StringUtils.concat(
                    String.valueOf(context.prefix()), "_m", mid.toString());

            FastSlowLock lock =
                context.kamaji().lockManager().acquire(
                    key,
                    new FastSlowLock(context.zooQueueId(), key));

            locks.add(
                new LockCallback(
                    multiCallback.newCallback(),
                    lock,
                    context));
        }
        sb.setCharAt(sb.length() - 1, ')');

        // do not send  requests inside loop,
        // we could acuire lock and throw exception after that,
        // holding lock forever
        locks.forEach(c -> {
            if (c.lock.lock(
                new LockRequest(LockerType.DELETE, context.zooQueueId(), c),
                context)
                != LockAcquireStatus.CALLBACK)
            {
                c.completed(null);
            }
        });

        locksCallback.request(sb);
        multiCallback.done();
    }

    private static final class LocksCallback
        implements FutureCallback<List<Object>>
    {
        private final ChangeContext context;
        private final DeleteIndexCallback callback;
        private StringBuilder request;

        private LocksCallback(
            final ChangeContext context,
            final DeleteIndexCallback callback)
        {
            this.context = context;
            this.callback = callback;
        }

        public LocksCallback request(final StringBuilder request) {
            this.request = request;
            return this;
        }

        @Override
        public void completed(final List<Object> objects) {
            AsyncClient client = context.kamaji().backendClient()
                .adjustZooHeaders(context.session().context())
                .addHeader(
                    YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
                    Long.toString(context.operationDateMillis()));

            client.execute(
                context.kamaji().backendHost(),
                new BasicAsyncRequestProducerGenerator(new String(request)),
                EmptyAsyncConsumerFactory.ANY_GOOD,
                context.session().listener().createContextGeneratorFor(client),
                callback);
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }

    private static final class LockCallback
        implements FutureCallback<LockAcquireStatus>
    {
        private final FutureCallback<Object> callback;
        private final FastSlowLock lock;
        private final ChangeContext context;
        private boolean lockReleased = false;
        private boolean holdingLock = false;

        private LockCallback(
            final FutureCallback<Object> callback,
            final FastSlowLock lock,
            final ChangeContext context)
        {
            this.callback = callback;
            this.lock = lock;
            this.context = context;
        }

        @Override
        public void completed(final LockAcquireStatus o) {
            synchronized (this) {
                holdingLock = true;
            }

            this.callback.completed(o);
        }

        public synchronized void releaseLock(final Exception e) {
            if (holdingLock) {
                lock.unlock(e, context);

                holdingLock = false;
            }

            if (!lockReleased) {
                context.kamaji().lockManager().release(lock.key());
                lockReleased = true;
            }
        }

        @Override
        public void failed(final Exception e) {
            this.callback.failed(e);
        }

        @Override
        public void cancelled() {
            this.callback.cancelled();
        }
    }

    private static final class DeleteIndexCallback
        extends BasicIndexCallback
    {
        private final Collection<LockCallback> lockCallbacks;

        private DeleteIndexCallback(
            final ChangeContext context,
            final Collection<LockCallback> lockCallbacks)
            throws JsonUnexpectedTokenException
        {
            super(context);

            this.lockCallbacks = lockCallbacks;
        }

        @Override
        public void completed(final Object result) {
            super.completed(result);

            lockCallbacks.forEach(r -> r.releaseLock(null));
        }

        @Override
        public void failed(final Exception e) {
            super.failed(e);

            lockCallbacks.forEach(r -> r.releaseLock(e));
        }

        @Override
        public void cancelled() {
            super.cancelled();

            lockCallbacks.forEach(r -> r.releaseLock(null));
        }
    }
}

