package ru.yandex.search.mail.kamaji;

import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;

public class FixDocsPrefixHandler implements ProxyRequestHandler {
    private static final String PREFIX = "&prefix=";
    private static final String URL = "url <";
    private static final int DEFAULT_IO_PRIO = 2000;
    private final Kamaji kamaji;

    public FixDocsPrefixHandler(final Kamaji kamaji) {
        this.kamaji = kamaji;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        FixSession fixSession = new FixSession(session);
        requestPrintKeys(fixSession);
    }

    private void requestPrintKeys(final FixSession session) {
//        session.session.logger().info(
//            "requestPrintKeys.offset: " + session.offset);
        AsyncClient client =
            kamaji.searchClient().adjust(session.session.context());
        client.execute(
            kamaji.config().searchConfig().host(),
            new BasicAsyncRequestProducerGenerator(session.printKeysRequest()),
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING,
            session.session.listener().createContextGeneratorFor(client),
            new PrintKeysCallback(session));
    }

    private void updatePrefixes(
        final FixSession session,
        final Set<Long> prefixes)
    {
        if (prefixes.size() == 0) {
            session.session.logger().info(
                "Finished batch processing, requesting new batch");
            requestPrintKeys(session);
            return;
        }
        session.session.logger().info(
            "batch prefixes left: " + prefixes.size());
        AsyncClient client =
            kamaji.backendClient().adjust(session.session.context());
        Iterator<Long> iter = prefixes.iterator();
        long prefix = iter.next();
        iter.remove();
        String updateJson =
            "{\"prefix\":" + prefix
            + ", \"query\":\"url:" + session.urlPrefix
                + Long.toString(prefix) + "_*\""
            + ", \"docs\":[{\"__prefix\":\"" + prefix + '\"'
                + ",\"version\":\"" + session.version + "\"}]}";
        session.session.logger().info("updating prefix: " + prefix + ": "
            + updateJson);
        String uri = "/update?kamaji-fixer&url-prefix=" + session.urlPrefix
            + "&batchSize=" + session.batchSize + "&left=" + prefixes.size()
            + PREFIX + prefix;
        client.execute(
            kamaji.config().backendConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                uri,
                updateJson,
                ContentType.APPLICATION_JSON),
            EmptyAsyncConsumerFactory.OK,
            session.session.listener().createContextGeneratorFor(client),
            new UpdatePrefixCallback(session, prefixes, prefix));
    }

    private final class UpdatePrefixCallback
        extends AbstractProxySessionCallback<Void>
    {
        private final FixSession fixSession;
        private final Set<Long> prefixes;
        private final long updatedPrefix;

        UpdatePrefixCallback(
            final FixSession fixSession,
            final Set<Long> prefixes,
            final long updatedPrefix)
        {
            super(fixSession.session);
            this.fixSession = fixSession;
            this.prefixes = prefixes;
            this.updatedPrefix = updatedPrefix;
        }

        @Override
        public void completed(final Void v) {
            fixSession.session.logger()
                .info("Updated prefix: " + updatedPrefix);
            updatePrefixes(fixSession, prefixes);
        }
    }

    private final class PrintKeysCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final FixSession fixSession;

        private PrintKeysCallback(final FixSession fixSession) {
            super(fixSession.session);
            this.fixSession = fixSession;
        }

        @Override
        public void completed(final JsonObject o) {
            try {
//                session.session.logger().fine(
//                    "response: "
//                    + ru.yandex.json.writer.JsonType.HUMAN_READABLE
//                        .toString(o));
                JsonMap root = o.asMap();
                if (root.keySet().size() == 0) {
                    fixSession.session.logger().info(
                        "Empty printkeys results, finished processing");
                    fixSession.session.response(HttpStatus.SC_OK);
                    return;
                }
                final String urlPrefix = fixSession.urlPrefix;
                int prefixLen = urlPrefix.length();
                final LinkedHashSet<Long> prefixes = new LinkedHashSet<>();
                for (String key: root.keySet()) {
                    if (!key.startsWith(urlPrefix)) {
                        fixSession.session.logger().warning("url key <" + key
                            + "> doesn't starts with prefix <"
                            + urlPrefix + '>');
                        if (prefixes.size() == 0) {
                            fixSession.offset++;
                        }
                        continue;
                    }
                    if (key.length() <= prefixLen + 2) {
                        fixSession.session.logger().warning(
                            "Can't extract prefix from "
                            + URL + key
                            + ">: url to short");
                        if (prefixes.size() == 0) {
                            fixSession.offset++;
                        }
                        continue;
                    }
                    int sep = key.indexOf('_', prefixLen + 1);
                    if (sep == -1) {
                        fixSession.session.logger().warning(
                            "Can't  extract prefix from "
                            + URL + key
                            + ">: enclosing '_' not found");
                        if (prefixes.size() == 0) {
                            fixSession.offset++;
                        }
                        continue;
                    }
                    String prefixStr = key.substring(prefixLen, sep);
                    try {
                        long prefix = Long.parseLong(prefixStr);
                        prefixes.add(prefix);
                    } catch (NumberFormatException e) {
                        fixSession.session.logger().log(
                            Level.WARNING,
                            "Can't extract  prefix from "
                            + URL + key
                            + '>',
                            e);
                        if (prefixes.size() == 0) {
                            fixSession.offset++;
                        }
                        continue;
                    }
                }
                if (prefixes.size() == 0) {
                    //all prefixes fetched are invalid
                    //retry with increased offset
                    requestPrintKeys(fixSession);
                    return;
                } else {
                    updatePrefixes(fixSession, prefixes);
                }
            } catch (JsonException e) {
                failed(e);
            }
        }
    }

    private static final class FixSession {
        private final ProxySession session;
        private final String urlPrefix;
        private final long version;
        private final int batchSize;
        private final int shard;
        private final int prio;
        private final String printKeysRequest;
        private long offset = 0;

        FixSession(final ProxySession session) throws BadRequestException {
            this.session = session;
            urlPrefix = session.params().getString("url-prefix");
            version = session.params().getLong("version");
            batchSize = session.params().getInt("batch-size");
            shard = session.params().getInt("shard");
            prio = session.params().getInt("IO_PRIO", DEFAULT_IO_PRIO);

            printKeysRequest = "/printkeys?"
                + "user=" + shard
                + "&json-type=dollar"
                + PREFIX + urlPrefix
                + "&field=url"
                + "&skip-deleted"
                + "&IO_PRIO=" + prio
                + "&text=url:" + urlPrefix + "*+AND+NOT+version:" + version
                + "&length=" + batchSize;
        }

        public String printKeysRequest() {
            if (offset == 0) {
                return printKeysRequest;
            } else {
                return printKeysRequest + "&offset=" + offset;
            }
        }
    }
}
