package ru.yandex.search.mail.kamaji;

import java.io.IOException;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.util.EnumMap;
import java.util.Map;

import org.apache.http.Header;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.client.wmi.FilterSearchErrorClassifier;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.LockStorage;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.proxy.StaticProxyHandler;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumerFactory;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock;
import ru.yandex.search.mail.kamaji.lock.TikaiteMemoryLimiter;
import ru.yandex.search.mail.kamaji.usertype.UserTypeHandler;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.StatsPair;
import ru.yandex.stater.StatsPairStaterFactory;

public class Kamaji
    extends HttpProxy<ImmutableKamajiConfig>
    implements HttpAsyncRequestHandler<Object>
{
    public static final int IOPRIO = 3000;

    private static final Long ZERO = 0L;
    private static final Long ONE = 1L;

    private final Map<ChangeType, ChangeHandler> changeHandlers =
        new EnumMap<>(ChangeType.class);
    private final AsyncClient backendClient;
    private final AsyncClient searchClient;
    private final AsyncClient slowIndexerClient;
    private final HttpHost backendHost;
    private final HttpHost slowIndexerHost;
    private final BlackboxClient blackboxClient;
    private final AsyncClient tikaiteClient;
    private final HttpHost tikaiteHost;
    private final Tvm2TicketRenewalTask tvm2RenewalTask;
    private final AsyncClient filterSearchClient;
    private final String filterSearchUri;
    private final UpstreamStater tikaiteStater;
    private final TimeFrameQueue<Long> tikaiteRequests;
    private final UpstreamStater filterSearchStater;
    private final TimeFrameQueue<StatsPair<Long, Long>> firstMail;
    private final LockStorage<String, FastSlowLock> lockManager;
    private final TimeFrameQueue<Long> lockStuckTime;
    private final TimeFrameQueue<Integer> inboxCorpTextLength;
    private final TimeFrameQueue<Integer> inboxPddTextLength;
    private final TimeFrameQueue<Integer> inboxWebTextLength;
    private final TimeFrameQueue<Integer> sentCorpTextLength;
    private final TimeFrameQueue<Integer> sentPddTextLength;
    private final TimeFrameQueue<Integer> sentWebTextLength;
    private final TikaiteMemoryLimiter tikaiteMemoryLimiter;

    // CSOFF: MethodLength
    public Kamaji(final ImmutableKamajiConfig config)
        throws GeneralSecurityException,
            HttpException,
            IOException,
            JsonException,
            URISyntaxException
    {
        super(config);
        this.lockManager = new LockStorage<>();

        if (config.tikaiteMemoryLimit() > 0) {
            tikaiteMemoryLimiter = new TikaiteMemoryLimiter(this);
        } else {
            tikaiteMemoryLimiter = null;
        }

        backendClient = client("Backend", config.backendConfig());
        searchClient = client("Search", config.searchConfig());
        slowIndexerClient =
            client("Slow-Indexer", config.slowIndexerConfig());

        backendHost = config.backendConfig().host();
        slowIndexerHost = config.slowIndexerConfig().host();
        blackboxClient =
            registerClient(
                "Blackbox",
                new BlackboxClient(reactor, config.blackboxConfig()),
                config.blackboxConfig());
        tikaiteClient = client("Tikaite", config.tikaiteConfig());
        tikaiteHost = config.tikaiteConfig().host();
        tvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("tvm2"),
            serviceContextRenewalTask,
            config.tvm2ClientConfig());
        filterSearchClient = client(
            "FilterSearch",
            config.filterSearchConfig(),
            FilterSearchErrorClassifier.INSTANCE);
        filterSearchUri =
            config.filterSearchConfig().uri().toASCIIString()
            + config.filterSearchConfig().firstCgiSeparator()
            + "order=default&full_folders_and_labels=1&uid=";
        tikaiteStater =
            new UpstreamStater(config.metricsTimeFrame(), "tikaite");
        tikaiteRequests =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        filterSearchStater = new UpstreamStater(
            config.metricsTimeFrame(),
            "filter-search");
        firstMail = new TimeFrameQueue<>(config.metricsTimeFrame());
        lockStuckTime = new TimeFrameQueue<>(config.metricsTimeFrame());
        inboxCorpTextLength =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        inboxPddTextLength =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        inboxWebTextLength =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        sentCorpTextLength =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        sentPddTextLength =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        sentWebTextLength =
            new TimeFrameQueue<>(config.metricsTimeFrame());

        changeHandlers.put(ChangeType.STORE, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.REINDEX, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.UPDATE, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.MOVE, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.MOVE_TO_TAB, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.COPY, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.THREADS_JOIN, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.UPDATE_ATTACH, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.QUICK_SAVE, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.SEARCH_UPDATE, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.FIELDS_UPDATE, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.SYNC_STORE, UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.SYNC_UPDATE, UpdateHandler.INSTANCE);
        changeHandlers.put(
            ChangeType.SYNC_THREADS_JOIN,
            UpdateHandler.INSTANCE);
        changeHandlers.put(
            ChangeType.MESSAGE_STORAGE_CHANGE,
            UpdateHandler.INSTANCE);
        changeHandlers.put(ChangeType.DELETE, DeletePgHandler.INSTANCE);
        changeHandlers.put(ChangeType.SYNC_DELETE, DeletePgHandler.INSTANCE);
        changeHandlers.put(ChangeType.USER_DELETE, new DeleteUserPgHandler());
        changeHandlers.put(
            ChangeType.SEARCH_MIDS_CLEANUP,
            MidsCleanupHandler.INSTANCE);
        changeHandlers.put(
            ChangeType.USER_TYPE_UPDATE,
            UserTypeHandler.INSTANCE);

        register(
            new Pattern<>("/notify", false),
            this,
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/tikaite", false),
            new TikaiteProxyHandler(this));
        StaticProxyHandler luceneProxyHandler =
            new StaticProxyHandler(backendClient, backendHost);
        register(new Pattern<>("/getQueueId", false), luceneProxyHandler);
        register(new Pattern<>("/queue/", true), luceneProxyHandler);
        register(new Pattern<>("/ping", false), new KamajiPingHandler(this));
        register(
            new Pattern<>("/fix-docs-prefix", false),
            new FixDocsPrefixHandler(this));
        register(
            new Pattern<>("/subscriptions/move_from_furita", false),
            new KamajiInitialMoveSubscriptions(this));
        register(new Pattern<>("", true), luceneProxyHandler);

        registerStater(
            new PassiveStaterAdapter<>(
                firstMail,
                new StatsPairStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "first-mail-count_ammm",
                        IntegralSumAggregatorFactory.INSTANCE
                    ),
                    new NamedStatsAggregatorFactory<>(
                        "first-mail-errors_ammm",
                        IntegralSumAggregatorFactory.INSTANCE
                    ))));

        registerStater(
            new PassiveStaterAdapter<>(
                lockStuckTime,
                    new NamedStatsAggregatorFactory<>(
                        "lock-stuck-time_axxx",
                        new MaxAggregatorFactory(0)
                    )));

        registerStater(tikaiteStater);
        registerStater(
            new PassiveStaterAdapter<>(
                tikaiteRequests,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "tikaite-failed-requests_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "tikaite-total-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));

        registerStater(
            new PassiveStaterAdapter<>(
                inboxCorpTextLength,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "inbox-corp-text-length_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "inbox-corp-text-length-count_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                inboxPddTextLength,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "inbox-pdd-text-length_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "inbox-pdd-text-length-count_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                inboxWebTextLength,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "inbox-web-text-length_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "inbox-web-text-length-count_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                sentCorpTextLength,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "sent-corp-text-length_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "sent-corp-text-length-count_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                sentPddTextLength,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "sent-pdd-text-length_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "sent-pdd-text-length-count_ammm",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                sentWebTextLength,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "sent-web-text-length_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "sent-web-text-length-count_ammm",
                        CountAggregatorFactory.INSTANCE))));

        registerStater(filterSearchStater);
    }
    // CSON: MethodLength

    public LockStorage<String, FastSlowLock> lockManager() {
        return lockManager;
    }

    public AsyncClient backendClient() {
        return backendClient;
    }

    public AsyncClient searchClient() {
        return searchClient;
    }

    public AsyncClient slowIndexerClient() {
        return slowIndexerClient;
    }

    public HttpHost backendHost() {
        return backendHost;
    }

    public HttpHost slowIndexerHost() {
        return slowIndexerHost;
    }

    public BlackboxClient blackboxClient() {
        return blackboxClient;
    }

    public AsyncClient tikaiteClient() {
        return tikaiteClient;
    }

    public HttpHost tikaiteHost() {
        return tikaiteHost;
    }

    public String blackboxTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.blackboxTvmClientId());
    }

    public String tikaiteTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.tikaiteTvmClientId());
    }

    public String unistorageTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.unistorageTvmClientId());
    }

    public String filterSearchTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.filterSearchTvmClientId());
    }

    public AsyncClient filterSearchClient() {
        return filterSearchClient;
    }

    public String filterSearchUri() {
        return filterSearchUri;
    }

    @Override
    public void start() throws IOException {
        tvm2RenewalTask.start();
        super.start();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        if (tikaiteMemoryLimiter != null) {
            tikaiteMemoryLimiter.status(status, verbose);
        }

        return status;
    }

    @Override
    public void close() throws IOException {
        if (tikaiteMemoryLimiter != null) {
            tikaiteMemoryLimiter.close();
        }
        tvm2RenewalTask.cancel();
        super.close();
    }

    // ProxyRequestHandler implementation
    @Override
    public String toString() {
        return "Handles notification messages concerning user mailbox changes";
    }

    @Override
    public void handle(
        final Object payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(this, exchange, context);
        String mdb = session.params().getString(OracleFields.MDB);
        if (!"pg".equals(mdb)) {
            session.logger().fine(
                mdb + " now we support only pg");
            session.response(HttpStatus.SC_OK);
            return;
        }
        try {
            handlePgRequest(session, ValueUtils.asMap(payload));
        } catch (JsonException e) {
            throw new BadRequestException(
                "Failed to parse request: " + payload,
                e);
        }
    }

    private void handlePgRequest(
        final ProxySession session,
        final Map<?, ?> json)
        throws HttpException, JsonException
    {
        ChangeContext context = new ChangeContext(this, session, json);
        ChangeHandler handler = changeHandlers.get(context.changeType());
        if (handler == null) {
            handler = UnknownChangeHandler.INSTANCE;
        }
        handler.handle(context);
    }

    @Override
    public HttpAsyncRequestConsumer<Object> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity());
    }

    public void sendTikaiteRequest(
        final KamajiIndexationContext context,
        final FastSlowLock lock)
    {
        if (tikaiteMemoryLimiter != null) {
            tikaiteMemoryLimiter.sendTikaiteRequest(context, lock);
        } else {
            sendTikaiteRequest(
                context.changeContext().session(),
                context.stid(),
                new TikaiteCallback(context, lock));
        }
    }

    public UpstreamStater tikaiteStater() {
        return tikaiteStater;
    }

    public void tikaiteCompleted() {
        tikaiteRequests.accept(ZERO);
    }

    public void tikaiteFailed() {
        tikaiteRequests.accept(ONE);
    }

    public UpstreamStater filterSearchStater() {
        return filterSearchStater;
    }

    public TimeFrameQueue<StatsPair<Long, Long>> firstMailStater() {
        return firstMail;
    }

    public TimeFrameQueue<Integer> inboxCorpTextLength() {
        return inboxCorpTextLength;
    }

    public TimeFrameQueue<Integer> inboxPddTextLength() {
        return inboxPddTextLength;
    }

    public TimeFrameQueue<Integer> inboxWebTextLength() {
        return inboxWebTextLength;
    }

    public TimeFrameQueue<Integer> sentCorpTextLength() {
        return sentCorpTextLength;
    }

    public TimeFrameQueue<Integer> sentPddTextLength() {
        return sentPddTextLength;
    }

    public TimeFrameQueue<Integer> sentWebTextLength() {
        return sentWebTextLength;
    }

    public boolean updatePosition(
        final HttpRequest request,
        final HttpContext context,
        final FutureCallback<Object> callback)
    {
        Header queueIdHeader =
            request.getFirstHeader(YandexHeaders.ZOO_QUEUE_ID);
        Header shardIdHeader =
            request.getFirstHeader(YandexHeaders.ZOO_SHARD_ID);
        Header queueHeader =
            request.getFirstHeader(YandexHeaders.ZOO_QUEUE);

        if (shardIdHeader == null
            || queueHeader == null
            || queueIdHeader == null)
        {
            return false;
        }

        String queue = queueHeader.getValue();
        String shard = shardIdHeader.getValue();
        String position = queueIdHeader.getValue();

        final String uri = "/delete?updatePosition&prefix="
            + shard
            + "&shard=" + shard
            + "&service=" + queue
            + "&position=" + position;
        final String content = "{\"prefix\":" + shard
            + ",\"docs\":[]}";

        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(uri, content);
        AsyncClient client = backendClient.adjustZooHeaders(context);

        client.execute(
            backendHost,
            post,
            EmptyAsyncConsumerFactory.ANY_GOOD,
            callback);

        return true;
    }

    public void sendTikaiteRequest(
        final ProxySession session,
        final String stid,
        final FutureCallback<Object> callback)
    {
        AsyncClient client = tikaiteClient().adjust(session.context());
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(
                "/mail/handler?json-type=dollar&stid=" + stid);
        producerGenerator.addHeader(
            YandexHeaders.X_YA_SERVICE_TICKET,
            tikaiteTvm2Ticket());
        producerGenerator.addHeader(
            YandexHeaders.X_SRW_SERVICE_TICKET,
            unistorageTvm2Ticket());
        client.execute(
            tikaiteHost(),
            producerGenerator,
            JsonAsyncDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                callback,
                tikaiteStater()));
    }

    public void lockTime(final long lockTimeMs) {
        lockStuckTime.accept(lockTimeMs);
    }
}

