package ru.yandex.search.mail.kamaji;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.function.Consumer;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.function.CorpUidPredicate;
import ru.yandex.function.NullConsumer;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.mail.kamaji.update.IndexationContext;

public class KamajiIndexationContext implements IndexationContext {
    //for subscriptions, set inside kamaji
    public static final String KAMAJI_MAIL_CLICKS = "kamaji-mail-clicks";

    private static final EnumParser<FolderType> FOLDER_TYPE_ENUM_PARSER =
        new EnumParser<>(FolderType.class);

    private static final int MAIL_TYPE_CORP = 0;
    private static final int MAIL_TYPE_PDD = 1;
    private static final int MAIL_TYPE_WEB = 2;

    private final ChangeContext changeContext;
    private final AsyncClient indexClient;
    private final MailMetaInfo meta;
    private final FutureCallback<Object> callback;
    private final Map<String, String> auxillary;
    private final String mid;
    private final String stid;
    private final BlackboxUserinfo userinfo;
    private final Consumer<? super Integer> textLengthConsumer;
    private final FolderType folderType;
    private final boolean isSpamOrTrash;

    public KamajiIndexationContext(
        final ChangeContext changeContext,
        final AsyncClient indexClient,
        final MailMetaInfo meta,
        final BlackboxUserinfo userinfo,
        final Map<String, String> auxillary,
        final FutureCallback<Object> callback)
        throws JsonUnexpectedTokenException, NotFoundException
    {
        this.changeContext = changeContext;
        this.indexClient = indexClient;
        this.auxillary = auxillary;
        this.userinfo = userinfo;

        this.meta = meta;
        this.callback = callback;

        String folderTypeStr = meta.get(MailIndexFields.FOLDER_TYPE);
        FolderType folderType = null;
        try {
            folderType =
                FOLDER_TYPE_ENUM_PARSER.apply(folderTypeStr);
        } catch (IllegalArgumentException | NullPointerException e) {
            changeContext.session().logger().log(
                Level.WARNING,
                "Unrecognized folder type " + folderTypeStr,
                e);
        }

        this.folderType = folderType;
        isSpamOrTrash =
            folderType == FolderType.SPAM
                || folderType == FolderType.TRASH
                || folderType == FolderType.HIDDEN_TRASH;

        mid = ValueUtils.asString(meta.get(MailMetaInfo.MID));
        stid = ValueUtils.asString(meta.get(MailMetaInfo.STID));
        Consumer<? super Integer> textLengthConsumer = NullConsumer.INSTANCE;
        if (meta.messageTypes().contains(MessageType.PEOPLE.typeNumber())
            && mid.equals(meta.get(MailMetaInfo.THREAD_ID_FIELD)))
        {
            if (changeContext.changeType() == ChangeType.STORE
                || changeContext.changeType() == ChangeType.SYNC_STORE)
            {
                int mailType;
                if (changeContext.prefix() < CorpUidPredicate.CORP_UID_BEGIN) {
                    mailType = MAIL_TYPE_WEB;
                } else if (changeContext.prefix() < CorpUidPredicate.CORP_UID_END) {
                    mailType = MAIL_TYPE_CORP;
                } else {
                    mailType = MAIL_TYPE_PDD;
                }
                Kamaji kamaji = changeContext.kamaji();
                if (folderType == FolderType.USER
                    || folderType == FolderType.INBOX)
                {
                    if (mailType == MAIL_TYPE_CORP) {
                        textLengthConsumer = kamaji.inboxCorpTextLength();
                    } else if (mailType == MAIL_TYPE_PDD) {
                        textLengthConsumer = kamaji.inboxPddTextLength();
                    } else {
                        textLengthConsumer = kamaji.inboxWebTextLength();
                    }
                } else if (folderType == FolderType.SENT) {
                    if (mailType == MAIL_TYPE_CORP) {
                        textLengthConsumer = kamaji.sentCorpTextLength();
                    } else if (mailType == MAIL_TYPE_PDD) {
                        textLengthConsumer = kamaji.sentPddTextLength();
                    } else {
                        textLengthConsumer = kamaji.sentWebTextLength();
                    }
                }
            }
        }
        this.textLengthConsumer = textLengthConsumer;
    }

    public KamajiIndexationContext(
        final ChangeContext changeContext,
        final AsyncClient indexClient,
        final MailMetaInfo meta,
        final BlackboxUserinfo userinfo,
        final FutureCallback<Object> cb)
        throws JsonUnexpectedTokenException, NotFoundException
    {
        this(
            changeContext,
            indexClient,
            meta,
            userinfo,
            Collections.emptyMap(),
            cb);
    }

    public ChangeContext changeContext() {
        return changeContext;
    }

    @Override
    public MailMetaInfo meta() {
        return meta;
    }

    @Override
    public AsyncClient indexClient() {
        return indexClient;
    }

    public FutureCallback<Object> callback() {
        return callback;
    }

    public String mid() {
        return mid;
    }

    public String stid() {
        return stid;
    }

    @Override
    public String url(final String hid) {
        return changeContext.url(mid, hid);
    }

    @Override
    public Collection<String> preserveFields() {
        return changeContext.preserveFields();
    }

    @Override
    public Collection<String> preserveSlowFields() {
        return changeContext.preserveSlowFields();
    }

    public BlackboxUserinfo userInfo() {
        return userinfo;
    }

    @Override
    public long prefix() {
        return changeContext.prefix();
    }

    @Override
    public Map<?, ?> updates() {
        Map<String, Map<?, ?>> updates = changeContext.updates();
        if (updates.isEmpty()) {
            return Collections.emptyMap();
        }

        return updates.get(mid());
    }

    @Override
    public Map<String, String> auxillary() {
        return auxillary;
    }

    @Override
    public Consumer<? super Integer> textLengthConsumer() {
        return textLengthConsumer;
    }

    public FolderType folderType() {
        return folderType;
    }

    public boolean isSpamOrTrash() {
        return isSpamOrTrash;
    }
}

