package ru.yandex.search.mail.kamaji;

import java.io.IOException;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.search.mail.kamaji.subscriptions.IndexModule;
import ru.yandex.search.mail.kamaji.update.IndexationContext;

public final class KamajiIndexer {
    private KamajiIndexer() {
    }

    public static void indexDocument(final KamajiIndexationContext context) {
        indexDocument(context, context.callback());
    }

    public static void indexDocument(
        final KamajiIndexationContext context,
        final FutureCallback<Object> callback)
    {
        if (context.changeContext().slow()) {
            callback.completed(null);
        } else {
            StringBuilderWriter sbw = new StringBuilderWriter();
            List<IndexModule> defaultSubindexers =
                context.changeContext().dbSubIndexers().get(MailSearchDatabases.DEFAULT);

            try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
                writer.startObject();
                writeFastDocumentWithPrelude(writer, context);
                writer.endObject();

                if (defaultSubindexers != null) {
                    for (IndexModule module : defaultSubindexers) {
                        for (Map<String, Object> doc
                            : module.indexDocuments(context))
                        {
                            writer.value(doc);
                        }
                    }
                }

                writer.endArray();
                writer.endObject();
            } catch (IOException e) {
                callback.failed(new ServiceUnavailableException(e));
                return;
            }

            FutureCallback<Object> defaultDbCallback = callback;
            if ((context.changeContext().dbSubIndexers().size() > 1 && defaultSubindexers != null)
                || (context.changeContext().dbSubIndexers().size() > 0 && defaultSubindexers == null))
            {
                MultiFutureCallback<Object> mfcb = new MultiFutureCallback<>(callback);
                defaultDbCallback = mfcb.newCallback();
                for (Map.Entry<MailSearchDatabases, List<IndexModule>> entry
                    : context.changeContext().dbSubIndexers().entrySet())
                {
                    if (entry.getKey() == MailSearchDatabases.DEFAULT
                        || entry.getValue().size() == 0)
                    {
                        continue;
                    }

                    StringBuilderWriter dbbw = new StringBuilderWriter();
                    Set<String> preserveFields = new LinkedHashSet<>();

                    int docs = 0;
                    try (JsonWriter writer = JsonType.DOLLAR.create(dbbw)) {
                        writer.startObject();
                        writer.key("prefix");
                        writer.value(context.prefix());
                        writer.key("AddIfNotExists");
                        writer.value(true);
                        writer.key("docs");
                        writer.startArray();
                        for (IndexModule module: entry.getValue()) {
                            for (Map<String, Object> doc: module.indexDocuments(context)) {
                                writer.value(doc);
                                docs += 1;
                                preserveFields.addAll(module.preserveFields());
                            }
                        }
                        writer.endArray();
                        if (!preserveFields.isEmpty()) {
                            writer.key("PreserveFields");
                            writer.startArray();
                            for (String field: preserveFields) {
                                writer.value(field);
                            }
                            writer.endArray();
                        }
                        writer.endObject();
                    } catch (IOException e) {
                        callback.failed(new ServiceUnavailableException(e));
                        return;
                    }

                    if (docs == 0) {
                        context.changeContext().session().logger().warning(
                            "Database " + entry.getKey() + " 0 docs indexed");
                        continue;
                    }

                    context.indexClient().execute(
                        context.changeContext().kamaji().backendHost(),
                        new BasicAsyncRequestProducerGenerator(
                            "/update?" + context.changeContext().userTag()
                                + "&fast&mid=" + context.mid()
                                + "&db=" + entry.getKey().dbName(),
                            dbbw.toString(),
                            ContentType.APPLICATION_JSON.withCharset(
                                context.indexClient().requestCharset())),
                        EmptyAsyncConsumerFactory.OK,
                        context.changeContext().session().listener()
                            .createContextGeneratorFor(context.indexClient()),
                        mfcb.newCallback());
                }

                mfcb.done();
            }
            context.indexClient().execute(
                context.changeContext().kamaji().backendHost(),
                new BasicAsyncRequestProducerGenerator(
                    "/update?" + context.changeContext().userTag()
                        + "&fast&mid=" + context.mid(),
                    sbw.toString(),
                    ContentType.APPLICATION_JSON.withCharset(
                        context.indexClient().requestCharset())),
                EmptyAsyncConsumerFactory.OK,
                context.changeContext().session().listener()
                    .createContextGeneratorFor(context.indexClient()),
                defaultDbCallback);
        }
    }

    public static void writeUpdateFields(
        final JsonWriter writer,
        final IndexationContext context)
        throws IOException
    {
        try {
            for (Map.Entry<?, ?> entry: context.updates().entrySet()) {
                writer.key(ValueUtils.asString(entry.getKey()));
                writer.value(entry.getValue());
            }
        } catch (JsonUnexpectedTokenException jute) {
            throw new IOException(jute);
        }
    }

    public static void writeFastFields(
        final JsonWriter writer,
        final IndexationContext context)
        throws IOException
    {
        for (Map.Entry<String, String> entry
            : context.meta().toFastDocMap().entrySet())
        {
            String key = entry.getKey();
            String value = entry.getValue();
            writer.key(key);
            writer.value(value);
        }
    }

    public static void writePreserveFields(
        final JsonWriter writer,
        final IndexationContext context)
        throws IOException
    {
        writer.key("AddIfNotExists");
        writer.value(true);
        writer.key("PreserveFields");
        writer.startArray();
        for (String field: context.preserveFields()) {
            writer.value(field);
        }

        for (String field: context.preserveSlowFields()) {
            writer.value(field);
        }

        writer.endArray();
    }

    public static void writeFastDocumentWithPrelude(
        final JsonWriter writer,
        final IndexationContext context)
        throws IOException
    {
        writer.key("prefix");
        writer.value(context.prefix());
        writePreserveFields(writer, context);
        writer.key("docs");
        writer.startArray();
        writer.startObject();
        writeFastFields(writer, context);
        writeUpdateFields(writer, context);
    }
}
