package ru.yandex.search.mail.kamaji;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;

import ru.yandex.dbfields.SubscriptionsIndexField;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.mail.search.mail.MailSearchServices;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.mail.search.SubscriptionsFields;
import ru.yandex.search.json.fieldfunction.FieldFunctionException;
import ru.yandex.search.json.fieldfunction.MakeSetFunction;
import ru.yandex.search.json.fieldfunction.SumMapFunction;
import ru.yandex.util.string.StringUtils;

public class KamajiInitialMoveSubscriptions implements ProxyRequestHandler {
    private final Kamaji kamaji;

    public KamajiInitialMoveSubscriptions(final Kamaji kamaji) {
        this.kamaji = kamaji;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        Long prefix = session.params().getLong("uid", null);
        if (prefix == null) {
            prefix = session.params().getLong("prefix");
        }

        StringBuilder text = new StringBuilder();
        text.append(SubscriptionsIndexField.SUBS_MESSAGE_TYPES.fieldName());
        text.append(":(7 OR 13 OR 100)");
        long from = System.currentTimeMillis() / 1000 - 60 * 60 * 24 * 31 * 12;
        long to = Long.MAX_VALUE;

        text.append(" AND ");
        text.append(
            SubscriptionsIndexField.SUBS_RECEIVED_MONTH_PREFIXED.fieldName());
        text.append(":[");
        text.append(from);
        text.append(" TO ");
        text.append(to);
        text.append(']');

        QueryConstructor qc = new QueryConstructor("/search-subs-initial?");
        qc.append("prefix", prefix);
        qc.append("text", text.toString());
        qc.append(
            "get",
            "url,subs_email,subs_received_month,subs_last_received_date,subs_names,subs_received_types");
        qc.append("sort", "subs_last_received_date");
        AsyncClient client =
            kamaji.searchClient().adjust(session.context());
        client.execute(
            kamaji.config().searchConfig().host(),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING,
            session.listener().createContextGeneratorFor(client),
            new CurrentSubscriptionsCallback(session, prefix));
    }

    private class CurrentSubscriptionsCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Long uid;

        public CurrentSubscriptionsCallback(
            final ProxySession session,
            final Long uid)
        {
            super(session);
            this.uid = uid;
        }

        @Override
        public void completed(final JsonObject jsonObject) {
            try {
                JsonMap result = jsonObject.asMap();
                JsonList hits = result.getList("hitsArray");
                Map<String, SubscriptionEntry> map =
                    new LinkedHashMap<>(hits.size() << 1);

                for (JsonObject item: hits) {
                    JsonMap itemMap = item.asMap();
                    String email = itemMap.getString("subs_email");
                    long lrd =
                        itemMap.getLong(
                            SubscriptionsIndexField.SUBS_LAST_RECEIVED_DATE.fieldName(),
                            -1L);
                    map.computeIfAbsent(email, (e) -> new SubscriptionEntry(uid, email, lrd))
                        .update(itemMap);
                }

                if (map.size() == 0) {
                    session.response(HttpStatus.SC_OK);
                } else {
                    AsyncClient client =
                        kamaji.backendClient().adjust(session.context());

                    QueryConstructor qc = new QueryConstructor("/update?subs_initial");
                    qc.append("prefix", uid);
                    qc.append("service", MailSearchServices.SUBSCRIPTIONS_1.service());
                    qc.append("db", MailSearchDatabases.SUBSCRIPTIONS.dbName());

                    StringBuilderWriter sbWriter = new StringBuilderWriter();
                    try (JsonWriter writer = new JsonWriter(sbWriter)) {
                        writer.startObject();
                        writer.key("prefix");
                        writer.value(uid);
                        writer.key("AddIfNotExists");
                        writer.value(true);
                        writer.key("docs");
                        writer.startArray();
                        for (Map.Entry<String, SubscriptionEntry> entry: map.entrySet()) {
                            writer.value(entry.getValue());
                        }
                        writer.endArray();
                        writer.endObject();
                    } catch (IOException e) {
                        failed(e);
                        return;
                    }

                    client.execute(
                        kamaji.backendHost(),
                        new BasicAsyncRequestProducerGenerator(
                            qc.toString(),
                            sbWriter.toString(),
                            StandardCharsets.UTF_8),
                        EmptyAsyncConsumerFactory.OK,
                        session.listener()
                            .createContextGeneratorFor(client),
                        new IndexCallback(session));
                }
            } catch (JsonException | BadRequestException e) {
                failed(e);
            }
        }
    }

    private static final class SubscriptionEntry implements JsonValue {
        private final long lastReceivedDate;
        private final String email;
        private final Long uid;
        private final Set<String> names;
        private final Map<String, Long> receivedTypes;
        private final Map<String, Long> readTypes;

        public SubscriptionEntry(final Long uid, final String email, final long lastReceivedDate) {
            this.email = email;
            this.uid = uid;
            this.lastReceivedDate = lastReceivedDate;
            names = new LinkedHashSet<>();
            receivedTypes = new LinkedHashMap<>();
            readTypes = new LinkedHashMap<>();
        }

        public void update(final JsonMap map) throws JsonException {
            map.get("subs_names", new CollectionParser<>(String::trim, () ->names, '\n'));
            try {
                MakeSetFunction.addToSet(names, map.getString("subs_names", ""), 20);
                SumMapFunction.addToMap(this.receivedTypes, map.getString("subs_received_types", ""));
                SumMapFunction.addToMap(this.readTypes, map.getString("subs_read_types", ""));
            } catch (FieldFunctionException ffe) {
                throw new JsonException(ffe);
            }
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("url");
            writer.value("subs_" + this.uid + "_" + this.email);
            writer.key(SubscriptionsFields.SUBS_EMAIL.stored());
            writer.value(email);
            writer.key(SubscriptionsFields.SUBS_RECEIVED_DATE.stored());
            writer.value(lastReceivedDate);
            writer.key(SubscriptionsFields.SUBS_UID.stored());
            writer.value(uid);
            int sep = MailAliases.emailSeparatorPos(email);
            writer.key(SubscriptionsFields.SUBS_DOMAIN.stored());
            writer.value(MailAliases.domain(email, sep));
            writer.key(SubscriptionsFields.SUBS_RECEIVED_TYPES.stored());
            writer.value(SumMapFunction.mapToString(this.receivedTypes));
            writer.key(SubscriptionsFields.SUBS_READ_TYPES.stored());
            writer.value(SumMapFunction.mapToString(this.readTypes));
            writer.key(SubscriptionsFields.SUBS_NAMES.stored());
            writer.value(StringUtils.join(names, '\n'));
            writer.endObject();
        }
    }

    private static class IndexCallback extends AbstractProxySessionCallback<Object> {
        public IndexCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Object o) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
