package ru.yandex.search.mail.kamaji;

import org.apache.http.HttpHeaders;
import org.apache.http.entity.ContentType;

import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.PgFields;
import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.parser.uri.QueryParameter;
import ru.yandex.parser.uri.QueryParser;

public class OffsetCallback extends BasicIndexCallback {
    private final int nextOffset;
    private final int batchSize;
    private final AsyncClient client;
    private final boolean needCommit;

    public OffsetCallback(final ChangeContext context)
        throws BadRequestException, JsonUnexpectedTokenException
    {
        this(context, false);
    }

    public OffsetCallback(
        final ChangeContext context,
        final boolean useBatches)
        throws BadRequestException, JsonUnexpectedTokenException
    {
        super(context);
        if (useBatches) {
            int changedLeft = changed.size() - context.offset();
            if (changedLeft <= 0) {
                throw new BadRequestException(
                    "Malformed request: " + context.humanReadableJson()
                    + " with offset " + context.offset()
                    + " because changed size is " + changed.size());
            }
            int maxBatch =
                context.kamaji().config().filterSearchConfig().batchSize();
            if (changedLeft <= maxBatch) {
                batchSize = changedLeft;
                nextOffset = 0;
            } else {
                batchSize = maxBatch;
                nextOffset = context.offset() + batchSize;
            }
        } else {
            nextOffset = 0;
            batchSize = 1;
        }

        if (!context.slow()) {
            if (nextOffset == 0 && batchSize == 1) {
                needCommit = false;

                client = context.kamaji().backendClient()
                    .adjustZooHeaders(context.session().context());
            } else {
                needCommit = nextOffset == 0;

                client = context.kamaji().backendClient()
                    .adjustZooHeadersToCheck(context.session().context());
            }
        } else {
            client = context.kamaji().backendClient();
            needCommit = false;
        }
    }

    public AsyncClient indexClient() {
        return client;
    }

    public int batchSize() {
        return batchSize;
    }

    @Override
    public void completed(final Object result) {
        if (context.slow()) {
            context.session().response(YandexHttpStatus.SC_OK);
        } else {
            AsyncClient client = context.kamaji().slowIndexerClient().adjust(
                context.session().context());
            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(
                    context.session().request().getRequestLine().getUri()
                    + "&slow",
                    JsonType.DOLLAR.toString(context.json()),
                    ContentType.APPLICATION_JSON
                        .withCharset(client.requestCharset()));
            if (context.changeType() == ChangeType.SEARCH_UPDATE) {
                producerGenerator.addHeader(
                    YandexHeaders.X_PEACH_QUEUE,
                    "reindex");
            }
            client.execute(
                context.kamaji().slowIndexerHost(),
                producerGenerator,
                EmptyAsyncConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(client),
                new SlowCallback(context));
        }
    }

    private class SlowCallback extends AbstractIndexCallback {
        SlowCallback(final ChangeContext context) {
            super(context);
        }

        @Override
        public void completed(final Object result) {
            if (nextOffset == 0) {
                CommitCallback commitCallback = new CommitCallback(context);
                if (needCommit) {
                    AsyncClient client = context.kamaji().backendClient()
                        .adjustZooHeaders(context.session().context());
                    client.execute(
                        context.kamaji().backendHost(),
                        new BasicAsyncRequestProducerGenerator(
                            "/delete?commit&" + context.userTag()
                                + "&operation-id="
                                + context.json().get(PgFields.OPERATION_ID),
                            "{\"docs\":[],\"prefix\":" + context.prefix()
                                + '}',
                            ContentType.APPLICATION_JSON
                                .withCharset(client.requestCharset())),
                        EmptyAsyncConsumerFactory.ANY_GOOD,
                        context.session().listener()
                            .createContextGeneratorFor(client),
                        commitCallback);
                } else {
                    commitCallback.completed(null);
                }
            } else {
                logLag();
                StringBuilder uri = new StringBuilder();
                StringBuilderProcessorAdapter adapter =
                    new StringBuilderProcessorAdapter(uri);
                context.session().uri().path(null).processWith(adapter);
                uri.append('?');
                boolean offsetSet = false;
                boolean first = true;
                QueryParser parser = context.session().uri().queryParser(null);
                for (QueryParameter param: parser) {
                    if (first) {
                        first = false;
                    } else {
                        uri.append('&');
                    }
                    uri.append(param.name());
                    if (param.name().equals("offset")) {
                        offsetSet = true;
                        uri.append('=');
                        uri.append(nextOffset);
                    } else if (param.value() != QueryParser.NULL) {
                        uri.append('=');
                        param.value().processWith(adapter);
                    }
                }
                if (!offsetSet) {
                    if (!first) {
                        uri.append('&');
                    }
                    uri.append("offset=");
                    uri.append(nextOffset);
                }
                if (context.session().uri().hasFragment()) {
                    uri.append('#');
                    context.session().uri().fragment(null).processWith(adapter);
                }
                String location = new String(uri);
                context.session().logger().info(
                    "Redirecting "
                    + context.session().request().getRequestLine().getUri()
                    + " to " + location);
                context.session().getResponse().addHeader(
                    HttpHeaders.LOCATION,
                    location);
                context.session()
                    .response(YandexHttpStatus.SC_TEMPORARY_REDIRECT);
            }
        }
    }

    private static class CommitCallback extends AbstractIndexCallback {
        CommitCallback(final ChangeContext context) {
            super(context);
        }

        @Override
        public void completed(final Object result) {
            logLag();
            context.session().response(YandexHttpStatus.SC_OK);
        }
    }
}

