package ru.yandex.search.mail.kamaji;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock.LockAcquireStatus;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock.LockRequest;
import ru.yandex.search.mail.kamaji.lock.FastSlowLock.LockerType;
import ru.yandex.search.mail.kamaji.update.slow.SlowDocumentProvider;
import ru.yandex.search.mail.kamaji.update.slow.SlowLockCallback;
import ru.yandex.search.mail.kamaji.update.slow.TikaiteContext;

public class TikaiteCallback extends AbstractCallback<Object> {
    private static final SlowDocumentProvider DOCUMENT_PROVIDER =
        new SlowDocumentProvider();

    private final FastSlowLock lock;

    public TikaiteCallback(
        final KamajiIndexationContext context,
        final FastSlowLock lock)
    {
        super(context, context.callback());

        this.lock = lock;
    }

    public TikaiteCallback(
        final KamajiIndexationContext context,
        final FastSlowLock lock,
        final FutureCallback<Object> callback)
    {
        super(context, callback);

        this.lock = lock;
    }

    @Override
    public void completed(final Object result) {
        context.changeContext().kamaji().tikaiteCompleted();
        completedInternal(result);
    }

    private void completedInternal(final Object result) {
        AsyncClient client =
            context
                .changeContext()
                .kamaji()
                .backendClient()
                .adjust(context.changeContext().session().context());

        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        List<String> urls;
        try (JsonWriter writer =
                 new DollarJsonWriter(
                     new OutputStreamWriter(out, client.requestCharset())))
        {
            // write document to writer add receive urls of documents
            // were written
            urls = DOCUMENT_PROVIDER.apply(result, writer, context);
        } catch (IOException | JsonUnexpectedTokenException e) {
            callback().failed(e);
            return;
        }

        TikaiteContext tikaiteContext
            = new TikaiteContext(lock, context)
            .client(client)
            .out(out)
            .urls(urls);

        SlowLockCallback lockCallback =
            new SlowLockCallback(tikaiteContext, callback());

        LockAcquireStatus status =
            lock.lock(
                new LockRequest(
                    LockerType.SLOW,
                    context.changeContext().zooQueueId(),
                    lockCallback),
                context.changeContext());

        if (status == LockAcquireStatus.SKIP) {
            context.changeContext().session().logger().info(
                "LockSkip for " + context.mid());

            callback().completed(null);
        } else if (status == LockAcquireStatus.LOCKED) {
            lockCallback.completed(status);
        } else {
            context.changeContext().session().logger().info(
                "LockWait for " + context.mid());
        }
    }

    @Override
    public void failed(final Exception e) {
        if (RequestErrorType.temporaryFailure(e)
            || !AsyncClient.serverUnavailalbe(e))
        {
            callback().failed(e);
        } else {
            context.changeContext().kamaji().tikaiteFailed();
            context.changeContext().session().logger().log(
                Level.WARNING,
                "Failed to retrieve data from tikaite, skipping document: "
                + context.changeContext().session().listener().details(),
                e);
            completedInternal(e);
        }
    }
}

