package ru.yandex.search.mail.kamaji;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class TikaiteProxyHandler implements ProxyRequestHandler {
    private static final String ASTERISK = "*";

    private static final Set<String> ALL_FIELDS =
        Collections.singleton(ASTERISK);

    private static final CollectionParser<String, Set<String>, Exception>
        GET_FIELDS_PARSER =
        new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);

    private final Kamaji kamaji;

    public TikaiteProxyHandler(final Kamaji kamaji) {
        this.kamaji = kamaji;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        Set<String> fields =
            session.params().get("get", ALL_FIELDS, GET_FIELDS_PARSER);

        String hid = session.params().getString("hid", null);

        if (fields.isEmpty()) {
            throw new BadRequestException(
                "Get param should be * or comma separated values");
        }

        String stid = session.params().getString("stid");
        kamaji.sendTikaiteRequest(
            session,
            stid,
            new TikaiteCallback(session, hid, fields));
    }

    private static final class TikaiteCallback
        extends AbstractProxySessionCallback<Object>
    {
        private final String hid;
        private final Set<String> getFields;
        private final JsonType jsonType;

        private TikaiteCallback(
            final ProxySession session,
            final String hid,
            final Set<String> getFields)
            throws BadRequestException
        {
            super(session);

            this.hid = hid;
            this.getFields = getFields;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void completed(final Object o) {
            int status = HttpStatus.SC_OK;

            StringBuilderWriter sbw = new StringBuilderWriter();
            boolean asterisk = getFields.size() == 1
                && ASTERISK.equalsIgnoreCase(
                getFields.iterator().next());
            try (JsonWriter jw = this.jsonType.create(sbw)) {
                Map<?, ?> root = ValueUtils.asMap(o);
                List<?> docs = ValueUtils.asList(root.get("docs"));
                if (docs.size() == 0) {
                    status = HttpStatus.SC_NOT_FOUND;
                } else {
                    jw.startArray();
                    for (Object docObj: docs) {
                        Map<?, ?> doc = ValueUtils.asMap(docObj);
                        if (hid != null
                            && !hid.equalsIgnoreCase(
                                ValueUtils.asStringOrNull(
                                    doc.get(MailIndexFields.HID))))
                        {
                            continue;
                        }

                        if (asterisk) {
                            jw.value(doc);
                        } else {
                            jw.startObject();
                            for (String field: getFields) {
                                jw.key(field);
                                jw.value(doc.get(field));
                            }
                            jw.endObject();
                        }
                    }
                    jw.endArray();
                }
            } catch (JsonException | IOException e) {
                failed(e);
                return;
            }

            session.response(status, sbw.toString());
        }
    }
}
