package ru.yandex.search.mail.kamaji.lock;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.StringBuilderable;
import ru.yandex.search.mail.kamaji.ChangeContext;
import ru.yandex.util.string.StringUtils;

public class FastSlowLock implements StringBuilderable {
    private static final WaitlistAppendStatus EMPTY_SKIP =
        new WaitlistAppendStatus(false, Collections.emptyList());
    private static final WaitlistAppendStatus EMPTY_APPEND =
        new WaitlistAppendStatus(true, Collections.emptyList());

    public enum LockerType {
        FAST,
        SLOW,
        DELETE
    }

    public enum LockAcquireStatus {
        LOCKED,
        SKIP,
        CALLBACK
    }

    public enum LockStatus {
        FREE,
        LOCKED
    }

    private final List<LockRequest> waitList;
    private final String key;

    private LockStatus status = LockStatus.FREE;
    private long lockTime = 0L;
    private long queueId;

    public FastSlowLock(
        final long queueId,
        final String key)
    {
        this.queueId = queueId;
        this.key = key;

        this.waitList = new ArrayList<>();
    }

    public synchronized LockStatus status() {
        return status;
    }

    private WaitlistAppendStatus updateWaitList(final LockRequest request) {
        WaitlistAppendStatus result = null;
        if (request == null) {
            result = EMPTY_SKIP;
        }

        if (waitList.isEmpty()) {
            result = EMPTY_APPEND;
            waitList.add(request);
        }

        if (result != null) {
            return result;
        }

        boolean status = true;
        List<LockRequest> skipList = new ArrayList<>(waitList.size());
        if (request.lockerType() == LockerType.SLOW) {
            Iterator<LockRequest> iterator = waitList.iterator();
            while (iterator.hasNext()) {
                LockRequest item = iterator.next();
                if (item.queueId() > request.queueId()) {
                    status = false;
                    break;
                } else if (item.lockerType() == LockerType.SLOW) {
                    iterator.remove();
                    skipList.add(item);
                }
            }
        } else {
            Iterator<LockRequest> iterator = waitList.iterator();
            while (iterator.hasNext()) {
                LockRequest item = iterator.next();
                if (item.lockerType() == LockerType.SLOW
                    && item.queueId() < request.queueId())
                {
                    iterator.remove();
                    skipList.add(item);
                }
            }
        }

        if (status) {
            waitList.add(request);
        }

        return new WaitlistAppendStatus(status, skipList);
    }

    public LockAcquireStatus lock(
        final LockRequest request,
        final ChangeContext context)
    {
        LockAcquireStatus result;
        List<LockRequest> failList = null;
        StringBuilder logSb = new StringBuilder();

        long lockTime = -1L;
        synchronized (this) {
            if (request.lockerType() == LockerType.SLOW
                && request.queueId() < this.queueId)
            {
                context.session().logger().info(
                    "SkipLock,  " + this.toStringFast());
                return LockAcquireStatus.SKIP;
            }

            if (status == LockStatus.FREE) {
                logSb.append("SuccessLocked, ");

                this.queueId = request.queueId();
                this.status = LockStatus.LOCKED;
                this.lockTime = System.currentTimeMillis();

                result = LockAcquireStatus.LOCKED;
            } else {
                logSb.append("FailAlreadyLocked, ");
                lockTime = System.currentTimeMillis() - this.lockTime;
                WaitlistAppendStatus status = updateWaitList(request);

                result = status.status();
                failList = status.skipList;
            }

            toStringBuilder(logSb);
        }

        if (lockTime > 0) {
            context.kamaji().lockTime(lockTime);
        }

        context.session().logger().info(logSb.toString());

        if (failList != null) {
            context.session().logger().info("Skipping " + failList.size());
            for (LockRequest lr: failList) {
                lr.callback().completed(LockAcquireStatus.SKIP);
            }
        }

        return result;
    }

    @Override
    public synchronized void toStringBuilder(final StringBuilder sb) {
        sb.append(key);
        sb.append(',');
        sb.append(queueId);
        sb.append(',');
        sb.append(lockTime);
    }

    public void unlock(final ChangeContext context) {
        this.unlock(null, context);
    }

    public void unlock(final Exception e, final ChangeContext context) {
        StringBuilder sb = new StringBuilder("Unlocking ");

        LockRequest next = null;
        List<LockRequest> exceptionList = null;

        synchronized (this) {
            if (status != LockStatus.LOCKED) {
                throw new IllegalStateException("Lock is not held");
            }

            if (e == null && !waitList.isEmpty()) {
                if (waitList.size() > 1) {
                    Collections.sort(waitList);
                }

                next = waitList.remove(0);
                queueId = next.queueId();
            } else if (e != null) {
                exceptionList = new ArrayList<>(waitList);
            }

            if (next == null) {
                status = LockStatus.FREE;
                lockTime = 0L;
            } else {
                lockTime = System.currentTimeMillis();
            }

            toStringBuilder(sb);
        }

        context.session().logger().info(sb.toString());

        if (exceptionList != null) {
            for (LockRequest request: exceptionList) {
                request.callback().failed(e);
            }
        } else if (next != null) {
            context.session().logger().info(
                "LockTook switch " + toString() + " to " + next.toString()
                    + " waitList " + waitList.size());
            next.callback().completed(LockAcquireStatus.LOCKED);
        }
    }

    public String key() {
        return key;
    }

    public static class LockRequest implements Comparable<LockRequest> {
        private static final int ODD_MULTI = 31;

        private final FutureCallback<LockAcquireStatus> callback;
        private final LockerType lockerType;
        private final long queueId;

        public LockRequest(
            final LockerType lockerType,
            final long queueId,
            final FutureCallback<LockAcquireStatus> callback)
        {
            this.callback = callback;
            this.lockerType = lockerType;
            this.queueId = queueId;
        }

        public FutureCallback<LockAcquireStatus> callback() {
            return callback;
        }

        public LockerType lockerType() {
            return lockerType;
        }

        public long queueId() {
            return queueId;
        }

        @Override
        public int compareTo(final LockRequest o) {
            int result;
            if (this.lockerType == o.lockerType()) {
                result = Long.compare(this.queueId, o.queueId);
            } else if (this.lockerType == LockerType.SLOW) {
                result = 1;
            } else if (o.lockerType == LockerType.SLOW) {
                result = -1;
            } else {
                result = Long.compare(this.queueId, o.queueId);
            }

            return result;
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof LockRequest) {
                return this.compareTo((LockRequest) o) == 0;
            }

            return false;
        }

        @Override
        public int hashCode() {
            return Long.hashCode(queueId) + lockerType.hashCode() * ODD_MULTI;
        }

        @Override
        public String toString() {
            return StringUtils.concat(
                lockerType.name(),
                ',',
                String.valueOf(queueId));
        }
    }

    private static final class WaitlistAppendStatus {
        private final boolean added;
        private final List<LockRequest> skipList;

        private WaitlistAppendStatus(
            final boolean added,
            final List<LockRequest> skipList)
        {
            this.added = added;
            this.skipList = skipList;
        }

        private LockAcquireStatus status() {
            if (added) {
                return LockAcquireStatus.CALLBACK;
            } else {
                return LockAcquireStatus.SKIP;
            }
        }
    }
}
