package ru.yandex.search.mail.kamaji.lock;

import java.io.IOException;
import java.net.ConnectException;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.kamaji.ChangeContext;
import ru.yandex.search.mail.kamaji.Kamaji;
import ru.yandex.search.mail.kamaji.KamajiIndexationContext;
import ru.yandex.search.mail.kamaji.TikaiteCallback;
import ru.yandex.util.string.StringUtils;

/**
 * Workaround for tikaite results in memory fitting.
 */
public class TikaiteMemoryLimiter implements Runnable, GenericAutoCloseable<IOException> {
    // default 30mb
    private static final long DEFAULT_SIZE = 30000000L;
    private final Queue<DelayedTikaiteRequest> queue;
    private final long memoryLimit;
    private final AtomicLong acquiredMemory;
    private final Thread thread;
    private final PrefixedLogger logger;

    private volatile boolean stopped = false;

    public TikaiteMemoryLimiter(final Kamaji kamaji) {
        this.logger = kamaji.logger();
        this.acquiredMemory = new AtomicLong(0);
        this.memoryLimit = kamaji.config().tikaiteMemoryLimit();
        this.queue = new ArrayBlockingQueue<>(kamaji.config().tikaiteMemoryQueueCapacity());

        this.thread = new Thread(this, "KamajiTikaiteMemoryLimiter");
        this.thread.start();
    }

    public void status(final Map<String, Object> status, final boolean verbose) {
        status.put("TikaiteMemoryLimiterAcquired", acquiredMemory.get());
        status.put("TikaiteMemoryLimiterMemLimit", memoryLimit);
        status.put("TikaiteMemoryLimiterQueue", queue.size());
    }

    public void sendTikaiteRequest(
        final KamajiIndexationContext context,
        final FastSlowLock lock)
    {
        long size = context.meta().mailSize();
        if (size < 0) {
            size = DEFAULT_SIZE;
        }

        PrefixedLogger logger = context.changeContext().session().logger();
        if (size > memoryLimit) {
            String errorText =
                "Mail size is too big " + size
                    + " but total memory limit is " + memoryLimit;

            logger.log(
                Level.SEVERE,
                errorText);
            context.callback().failed(new Exception(errorText));
            return;
        }

        if (!sendOrReject(context, size, lock)) {
            if (queue.offer(new DelayedTikaiteRequest(context, size, lock))) {
                logger.warning("No memory limit left, putting in queue, size is " + queue.size());
            } else {
                context.callback().failed(new ConnectException("No memory limit left, and queue is full"));
            }
        }
    }

    @Override
    public void close() throws IOException {
        stopped = true;
    }

    protected boolean sendOrReject(
        final KamajiIndexationContext context,
        final long size,
        final FastSlowLock lock)
    {
        while (true) {
            long acquired = acquiredMemory.get();
            long newAcquired = size + acquired;
            if (newAcquired > memoryLimit) {
                return false;
            }

            if (acquiredMemory.compareAndSet(acquired, newAcquired)) {
                // we are locked
                break;
            }
        }

        context.changeContext().session().logger().info(
            "Memory lock acquired for " + context.mid());

        ChangeContext changeContext = context.changeContext();
        Kamaji kamaji = changeContext.kamaji();
        ProxySession session = changeContext.session();
        kamaji.sendTikaiteRequest(
            session,
            context.stid(),
            new TikaiteCallback(
                context,
                lock,
                new MemoryUnlockCallback(context, context.callback(), size)));
        return true;
    }

    @Override
    public void run() {
        try {
            while (!stopped) {
                synchronized (this) {
                    this.wait(500);
                }

                processQueue();
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING,"Stopping tikaite memory limiter", ie);
        }
    }

    protected void processQueue() {
        int total = queue.size();
        int procesed = 0;
        while (acquiredMemory.get() < memoryLimit && procesed < total) {
            DelayedTikaiteRequest item = queue.poll();

            if (item == null) {
                break;
            }

            procesed += 1;

            KamajiIndexationContext context = item.indexationContext();
            long passedInQueue = System.currentTimeMillis() - item.createTs();
            if (context.changeContext().session().cancelled()) {
                String message =
                    "Sessiong is cancelled, dropping from queue " + context.mid();
                context.changeContext().session().logger().warning(message);
                item.callback().cancelled();
            } else {
                if (!sendOrReject(context, context.meta().mailSize(), item.lock())) {
                    if (!queue.offer(item)) {
                        item.callback().failed(
                            new ServerException(
                                YandexHttpStatus.SC_TOO_MANY_REQUESTS,
                                "Rejecting, no place in queue for " + context.mid()));
                    }
                } else {
                    context.changeContext().session().logger().warning(
                        StringUtils.concat(
                            "Scheduled from queue", context.mid(), ' ', String.valueOf(passedInQueue)));
                }
            }
        }
    }

    private class MemoryUnlockCallback implements FutureCallback<Object> {
        private final KamajiIndexationContext context;
        private final FutureCallback<Object> callback;
        private final long size;

        public MemoryUnlockCallback(
            final KamajiIndexationContext context,
            final FutureCallback<Object> callback,
            final long size)
        {
            this.context = context;
            this.callback = callback;
            this.size = size;
        }

        @Override
        public void completed(final Object o) {
            context.changeContext().session().logger().info(
                "Releasing on completed memory lock for " + context.mid());
            acquiredMemory.addAndGet(-size);
            callback.completed(o);

            synchronized (TikaiteMemoryLimiter.this) {
                TikaiteMemoryLimiter.this.notify();
            }
        }

        @Override
        public void failed(final Exception e) {
            context.changeContext().session().logger().info(
                "Releasing on failed memory lock for " + context.mid());
            acquiredMemory.addAndGet(-size);
            callback.failed(e);

            synchronized (TikaiteMemoryLimiter.this) {
                TikaiteMemoryLimiter.this.notify();
            }
        }

        @Override
        public void cancelled() {
            context.changeContext().session().logger().info(
                "Releasing on cancelled memory lock for " + context.mid());
            acquiredMemory.addAndGet(-size);
            callback.cancelled();

            synchronized (TikaiteMemoryLimiter.this) {
                TikaiteMemoryLimiter.this.notify();
            }
        }
    }

    private static final class DelayedTikaiteRequest {
        private final KamajiIndexationContext indexationContext;
        private final long size;
        private final FastSlowLock lock;
        private final long createTs;

        public DelayedTikaiteRequest(
            final KamajiIndexationContext indexationContext,
            final long size,
            final FastSlowLock lock)
        {
            this.indexationContext = indexationContext;
            this.size = size;
            this.lock = lock;
            this.createTs = System.currentTimeMillis();
        }

        public KamajiIndexationContext indexationContext() {
            return indexationContext;
        }

        public long size() {
            return size;
        }

        public FutureCallback<Object> callback() {
            return indexationContext.callback();
        }

        public FastSlowLock lock() {
            return lock;
        }

        public long createTs() {
            return createTs;
        }
    }
}
