package ru.yandex.search.mail.kamaji.senders;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.james.mime4j.dom.address.Mailbox;

import ru.yandex.blackbox.BlackboxAddress;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.email.EmailParser;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.mail.senders.SenderInfo;
import ru.yandex.parser.mail.senders.SenderType;
import ru.yandex.parser.mail.senders.SendersContext;
import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.mail.kamaji.KamajiIndexationContext;
import ru.yandex.search.mail.kamaji.subscriptions.SlowIndexModule;

public abstract class AbstractSendersIndexerModule implements SlowIndexModule {
    public static final List<String> UID_FIELDS =
        Collections.unmodifiableList(
            Arrays.asList(
                MailIndexFields.SENDERS_UID,
                MailIndexFields.SENDERS_DOMAIN_UID));

    public static final List<String> DATA_FIELDS =
        Collections.unmodifiableList(
            Arrays.asList(
                MailIndexFields.SENDERS_LCN,
                MailIndexFields.SENDERS_LAST_CONTACTED,
                MailIndexFields.SENDERS_NAMES,
                MailIndexFields.SENDERS_MESSAGE_TYPES,
                MailIndexFields.SENDERS_MAIL_TYPE,
                MailIndexFields.SENDERS_MAIL_TYPES,
                MailIndexFields.SENDERS_RECEIVED_COUNT,
                MailIndexFields.SENDERS_SENDER_TYPE,
                MailIndexFields.SENDERS_SENT_COUNT,
                MailIndexFields.SENDERS_STORE_FOLDERS,
                MailIndexFields.SENDERS_FROM_READ_COUNT));

    protected static final String FUNCTION = "function";
    protected static final String GET = "get";
    protected static final String MAKE_SET = "make_set";
    protected static final String SUM_MAP = "sum_map";

    protected static final String MAX_SENDERS_NAMES = "100";
    protected static final String MAX_MESSAGE_TYPES = MAX_SENDERS_NAMES;

    private static final int FUNCTION_CAPACITY = 4;
    private static final long MILLIS = 1000L;
    private static final long MAX_TIME_DESYNC = 3600;

    private static final ThreadLocalEmailParser
        EMAIL_PARSER =
        new ThreadLocalEmailParser();

    public static List<Address> extractSender(final MailMetaInfo meta) {
        List<Address> froms = fromMeta(meta, MailMetaInfo.FROM);
        if (froms.isEmpty()) {
            froms = fromMeta(meta, MailMetaInfo.REPLY_TO_FIELD);
        }
        return froms;
    }

    public static List<SenderInfo> extractSenders(
        final MailMetaInfo meta,
        final List<?> docs)
    {
        MailMetaInfo sendersMeta = new MailMetaInfo(meta);
        try {
            if (!docs.isEmpty()) {
                Map<?, ?> doc = ValueUtils.asMap(docs.get(0));
                for (Map.Entry<?, ?> entry: doc.entrySet()) {
                    Object key = entry.getKey();
                    Object value = entry.getValue();
                    if (key instanceof String && value instanceof String) {
                        sendersMeta.set((String) key, (String) value);
                    }
                }
            }
        } catch (JsonUnexpectedTokenException e) {
            return Collections.emptyList();
        }
        List<SenderInfo> result =
            new SendersContext(
                Objects.toString(
                    sendersMeta.getLocal(MailMetaInfo.HEADERS),
                    "\n"))
                .extractSenders();
        Iterator<SenderInfo> iter = result.iterator();
        while (iter.hasNext()) {
            // It is already indexed with senders_names
            if (iter.next().type() == SenderType.FROM) {
                iter.remove();
            }
        }
        return result;
    }

    protected List<Map<String, Object>> indexIncoming(
        final KamajiIndexationContext context,
        final Addresses to,
        final Address from,
        final List<SenderInfo> senders)
    {
        context.changeContext().session().logger().info(
            "Senders incoming to " + to
            + " from " + from
            + ", additional addrs: " + senders);
        List<Map<String, Object>> documents =
            new ArrayList<>((senders.size() + 1) << 1);
        documents.add(createReceivedDocument(from, SenderType.FROM, context));
        documents.add(
            createReceivedDomainDocument(
                MailAliases.INSTANCE.extractAndNormalizeDomain(
                    from.normalized()),
                SenderType.FROM,
                context));
        for (SenderInfo senderInfo: senders) {
            SenderType type = senderInfo.type();
            String addr = senderInfo.email();
            documents.add(
                createReceivedDocument(new Address(addr), type, context));
            documents.add(
                createReceivedDomainDocument(
                    MailAliases.INSTANCE.extractAndNormalizeDomain(addr),
                    type,
                    context));
        }
        return documents;
    }

    protected List<Map<String, Object>> indexOutgoing(
        final KamajiIndexationContext context,
        final Addresses to,
        final Address from)
    {
        List<Map<String, Object>> documents =
            new ArrayList<>(to.size() << 1);
        Set<String> domains = new HashSet<>();
        for (Address toAddress: to) {
            documents.add(createSentDocument(toAddress, context));
            String domain = MailAliases.INSTANCE.extractAndNormalizeDomain(
                toAddress.normalized());
            if (domains.add(domain)) {
                documents.add(createSentDomainDocument(domain, context));
            }
        }

        return documents;
    }

    @Override
    public List<Map<String, Object>> indexDocuments(
        final KamajiIndexationContext context,
        final List<?> docs)
    {
        FolderType folderType = context.folderType();
        if (folderType == FolderType.DRAFT) {
            context.changeContext().session().logger().warning(
                "This is a draft, skipping senders");
            return Collections.emptyList();
        }

        List<Address> froms = extractSender(context.meta());
        if (froms.size() != 1) {
            context.changeContext().session().logger().warning(
                "Senders skipping, more than one from");
            return Collections.emptyList();
        }

        Address from = froms.get(0);

        Addresses userAddressList = new Addresses();
        userAddressList.addAll(fromBlackbox(context.userInfo().addressList()));
        Addresses to = buildToAddresses(context.meta());
        boolean outgoing = userAddressList.contains(from);
        boolean incoming = userAddressList.intersects(to);

        List<Map<String, Object>> documents = Collections.emptyList();
        if (outgoing && incoming) {
            to.removeAll(userAddressList);
            if (to.size() > 0) {
                if (folderType == FolderType.INBOX
                    || folderType == FolderType.USER)
                {
                    context.changeContext().session().logger().fine(
                        "User in to and from, based on folder "
                            + "type indexing as incoming");
                    documents = indexIncoming(
                        context,
                        to,
                        from,
                        extractSenders(context.meta(), docs));
                } else {
                    context.changeContext().session().logger().fine(
                        "User in to and from, based on folder type "
                        + folderType + " indexing as outgoing");
                    documents = indexOutgoing(context, to, from);
                }
            } else {
                // ok, user sending to himself or to linked accounts
                context.changeContext().session().logger().fine(
                    "Message from user to bonded email address,"
                        + " do not save senders");
            }
        } else if (outgoing) {
            documents = indexOutgoing(context, to, from);
        } else {
            if (!incoming) {
                context.changeContext().session().logger().warning(
                    "Not outgoing and not incoming it's bad, "
                    + "but probably just mailing list. User addresses: "
                    + userAddressList + ", to: " + to + ", from: " + from);
            }
            documents = indexIncoming(
                context,
                to,
                from,
                extractSenders(context.meta(), docs));
        }

        return documents;
    }

    @Override
    public Collection<String> preserveFields() {
        return DATA_FIELDS;
    }

    private static Addresses buildToAddresses(final MailMetaInfo meta) {
        Addresses to = new Addresses();
        to.addAll(fromMeta(meta, MailMetaInfo.TO));
        to.addAll(fromMeta(meta, MailMetaInfo.CC));
        to.addAll(fromMeta(meta, MailMetaInfo.BCC));
        return to;
    }

    protected static Map<String, Object> createFunction(
        final String functionName,
        final List<?> args)
    {
        Map<String, Object> function = new HashMap<>(FUNCTION_CAPACITY);
        function.put(FUNCTION, functionName);
        function.put("args", args);
        return function;
    }

    private static Map<String, Object> initDocument(final MailMetaInfo meta) {
        Map<String, Object> doc = new HashMap<>();
        String lcn = meta.get(MailMetaInfo.LCN);
        if (lcn != null) {
            doc.put(MailIndexFields.SENDERS_LCN, lcn);
        }
        String receivedDateString = meta.get(MailMetaInfo.RECEIVED_DATE);
        if (receivedDateString != null) {
            int receivedDate;
            try {
                receivedDate = Integer.parseInt(receivedDateString);
            } catch (NumberFormatException e) {
                receivedDate = 0;
            }
            if (receivedDate > 0) {
                long maxReceivedDate =
                    System.currentTimeMillis() / MILLIS + MAX_TIME_DESYNC;
                if (receivedDate <= maxReceivedDate) {
                    doc.put(
                        MailIndexFields.SENDERS_LAST_CONTACTED,
                        createFunction(
                            "max",
                            Arrays.asList(
                                createFunction(
                                    GET,
                                    Collections.singletonList(
                                        MailIndexFields.SENDERS_LAST_CONTACTED)
                                ),
                                receivedDateString)));
                }
            }
        }
        return doc;
    }

    protected static Map<String, Object> createBaseDocument(
        final Address address,
        final SenderType senderType,
        final MailMetaInfo meta)
    {
        Map<String, Object> doc = initDocument(meta);
        setBaseDocumentUrl(doc, address.normalized(), senderType, meta);

        String names = address.names();
        if (names != null) {
            doc.put(
                MailIndexFields.SENDERS_NAMES,
                createFunction(
                    MAKE_SET,
                    Arrays.asList(
                        createFunction(
                            GET,
                            Collections.singletonList(
                                MailIndexFields.SENDERS_NAMES)),
                        names,
                        MAX_SENDERS_NAMES)));
        }

        return doc;
    }

    protected static Map<String, Object> setBaseDocumentUrl(
        final Map<String, Object> doc,
        final String address,
        final SenderType senderType,
        final MailMetaInfo meta)
    {
        String uid = meta.get(MailMetaInfo.UID);
        doc.put(
            MailMetaInfo.URL,
            senderType.sendersPrefix() + uid + '_' + address);
        doc.put(MailIndexFields.SENDERS_UID, uid);
        doc.put(
            MailIndexFields.SENDERS_SENDER_TYPE,
            senderType.typeName());
        return doc;
    }

    protected static Map<String, Object> setDomainDocumentUrl(
        final Map<String, Object> doc,
        final String domain,
        final SenderType senderType,
        final MailMetaInfo meta)
    {
        String uid = meta.get(MailMetaInfo.UID);
        doc.put(
            MailMetaInfo.URL,
            senderType.sendersDomainPrefix() + uid + '_' + domain);
        doc.put(MailIndexFields.SENDERS_DOMAIN_UID, uid);
        doc.put(MailIndexFields.SENDERS_SENDER_TYPE, senderType.toString());

        return doc;
    }

    private static Map<String, Object> createBaseDomainDocument(
        final String domain,
        final SenderType senderType,
        final MailMetaInfo meta)
    {
        Map<String, Object> doc = initDocument(meta);
        setDomainDocumentUrl(doc, domain, senderType, meta);
        return doc;
    }

    protected void incrementCounter(
        final Map<String, Object> doc,
        final String counter)
    {
        doc.put(counter, Collections.singletonMap(FUNCTION, "inc"));
    }

    public Map<String, Object> createReceivedDocument(
        final Address address,
        final SenderType senderType,
        final KamajiIndexationContext context)
    {
        return createBaseDocument(address, senderType, context.meta());
    }

    public Map<String, Object> createReceivedDomainDocument(
        final String domain,
        final SenderType senderType,
        final KamajiIndexationContext context)
    {
        return createBaseDomainDocument(domain, senderType, context.meta());
    }

    public Map<String, Object> createSentDocument(
        final Address address,
        final KamajiIndexationContext context)
    {
        return createBaseDocument(address, SenderType.FROM, context.meta());
    }

    public Map<String, Object> createSentDomainDocument(
        final String domain,
        final KamajiIndexationContext context)
    {
        return createBaseDomainDocument(
            domain,
            SenderType.FROM,
            context.meta());
    }

    private static List<Address> fromBlackbox(
        final List<BlackboxAddress> addressList)
    {
        List<Address> result = new ArrayList<>(addressList.size());
        for (BlackboxAddress address: addressList) {
            if (address.validatedFlag()) {
                result.add(new Address(address.email()));
            }
        }
        return result;
    }

    private static List<Address> fromMeta(
        final MailMetaInfo meta,
        final String name)
    {
        List<Address> addresses = new ArrayList<>();
        String body = meta.get(MailMetaInfo.HDR + name);
        if (body != null) {
            List<Mailbox> mailboxList = EMAIL_PARSER.get().parseDecoded(body);
            for (Mailbox mailbox: mailboxList) {
                addresses.add(
                    new Address(
                        mailbox.getAddress(),
                        mailbox.getName()));
            }
        }
        return addresses;
    }

    private static class ThreadLocalEmailParser
        extends ThreadLocal<EmailParser>
    {
        @Override
        protected EmailParser initialValue() {
            return new EmailParser();
        }
    }
}

