package ru.yandex.search.mail.kamaji.senders;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.email.types.MessageTypeToString;
import ru.yandex.parser.mail.senders.SenderType;
import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.mail.kamaji.KamajiIndexationContext;

public class StoreSendersIndexerModule
    extends AbstractSendersIndexerModule
{
    public static final StoreSendersIndexerModule INSTANCE =
        new StoreSendersIndexerModule();

    public static final List<String> STORE_PRESERVE_FIELDS =
        Collections.unmodifiableList(
            Arrays.asList(
                MailIndexFields.SENDERS_FROM_READ_COUNT,
                MailIndexFields.SENDERS_LAST_CONTACTED,
                MailIndexFields.SENDERS_MAIL_TYPE,
                MailIndexFields.SENDERS_MAIL_TYPES,
                MailIndexFields.SENDERS_MESSAGE_TYPES,
                MailIndexFields.SENDERS_NAMES,
                MailIndexFields.SENDERS_RECEIVED_COUNT,
                MailIndexFields.SENDERS_SENDER_TYPE,
                MailIndexFields.SENDERS_SENT_COUNT,
                MailIndexFields.SENDERS_STORE_FOLDERS));

    private static final Set<Integer> MAIL_TYPES =
        Collections.unmodifiableSet(
            MessageType.MAIL_TYPES.stream()
                .map(MessageType::typeNumber)
                .collect(Collectors.toSet()));

    protected StoreSendersIndexerModule() {
    }

    @Override
    public Collection<String> preserveFields() {
        return STORE_PRESERVE_FIELDS;
    }

    @Override
    public Map<String, Object> createSentDocument(
        final Address address,
        final KamajiIndexationContext context)
    {
        Map<String, Object> doc =
            super.createSentDocument(address, context);
        incrementCounter(doc, MailIndexFields.SENDERS_SENT_COUNT);
        return doc;
    }

    @Override
    public Map<String, Object> createSentDomainDocument(
        final String domain,
        final KamajiIndexationContext context)
    {
        Map<String, Object> doc =
            super.createSentDomainDocument(domain, context);
        incrementCounter(doc, MailIndexFields.SENDERS_SENT_COUNT);
        return doc;
    }

    @Override
    public Map<String, Object> createReceivedDomainDocument(
        final String domain,
        final SenderType senderType,
        final KamajiIndexationContext context)
    {
        Map<String, Object> doc =
            super.createReceivedDomainDocument(domain, senderType, context);
        if (context.isSpamOrTrash()) {
            context.changeContext().session().logger().info(
                "Skipping domain received count update for "
                + context.folderType());
        } else {
            incrementCounter(doc, MailIndexFields.SENDERS_RECEIVED_COUNT);
        }

        return fillReceivedDoc(context, doc);
    }

    @Override
    public Map<String, Object> createReceivedDocument(
        final Address address,
        final SenderType senderType,
        final KamajiIndexationContext context)
    {
        MailMetaInfo meta = context.meta();
        Map<String, Object> doc =
            createBaseDocument(address, senderType, meta);
        return fillReceivedDoc(context, doc);
    }

    protected Map<String, Object> fillReceivedDoc(
        final KamajiIndexationContext context,
        final Map<String, Object> doc)
    {
        if (context.isSpamOrTrash()) {
            context.changeContext().session().logger().info(
                "Skipping received count update for " + context.folderType());
        } else {
            incrementCounter(doc, MailIndexFields.SENDERS_RECEIVED_COUNT);
        }

        Set<Integer> types = context.meta().messageTypes();
        StringBuilder sb = new StringBuilder();
        if (types != null && !types.isEmpty()) {
            for (Integer itype: types) {
                if (MAIL_TYPES.contains(itype)) {
                    sb.append(MessageTypeToString.INSTANCE.lookup(itype));
                    sb.append("\t1\n");
                }
            }

            final int mTypeMultiplier = 3;
            StringBuilder typesSet =
                new StringBuilder(types.size() * mTypeMultiplier);
            for (Integer type: types) {
                typesSet.append(type);
                typesSet.append('\n');
            }

            typesSet.setLength(typesSet.length() - 1);

            doc.put(
                MailIndexFields.SENDERS_MESSAGE_TYPES,
                createFunction(
                    MAKE_SET,
                    Arrays.asList(
                        createFunction(
                            GET,
                            Collections.singletonList(
                                MailIndexFields.SENDERS_MESSAGE_TYPES)),
                        typesSet.toString(),
                        MAX_MESSAGE_TYPES)));
        }

        if (sb.length() > 0) {
            sb.setLength(sb.length() - 1);
            Map<String, Object> mailTypes =
                createFunction(
                    SUM_MAP,
                    Arrays.asList(
                        createFunction(
                            GET,
                            Collections.singletonList(
                                MailIndexFields.SENDERS_MAIL_TYPES)),
                        sb.toString()));

            doc.put(
                MailIndexFields.SENDERS_MAIL_TYPES,
                mailTypes);

            doc.put(
                MailIndexFields.SENDERS_MAIL_TYPE,
                createFunction(
                    "map_max",
                    Collections.singletonList(mailTypes)));
        }

        FolderType folderType = context.folderType();

        if (folderType == null || folderType == FolderType.DRAFT) {
            return doc;
        }

        doc.put(
            MailIndexFields.SENDERS_STORE_FOLDERS,
            createFunction(
                SUM_MAP,
                Arrays.asList(
                    createFunction(
                        GET,
                        Collections.singletonList(
                            MailIndexFields.SENDERS_STORE_FOLDERS)),
                    folderType.toString().toLowerCase(Locale.ROOT) + "\t1")));

        return doc;
    }
}
