package ru.yandex.search.mail.kamaji.subscriptions;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.dbfields.ChangeType;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.ps.mail.search.SubscriptionsFields;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.mail.kamaji.ChangeContext;

public class NewSubscriptionsModuleFactory implements IndexModuleFactory {
    public static final Set<Integer> INDEXED_MESSAGE_TYPES =
        Collections.unmodifiableSet(
            new LinkedHashSet<>(Arrays.asList(7, 13, 100)));

    public static final NewSubscriptionsModuleFactory INSTANCE
        = new NewSubscriptionsModuleFactory();

    public static final Set<String> PRESERVE_FIELDS =
        Collections.unmodifiableSet(
            new HashSet<>(
                Arrays.asList(
                    SubscriptionsFields.SUBS_USER_ACTION_TS.stored(),
                    SubscriptionsFields.SUBS_HIDDEN_TYPES.stored(),
                    SubscriptionsFields.SUBS_READ_TYPES.stored(),
                    SubscriptionsFields.SUBS_RECEIVED_TYPES.stored(),
                    SubscriptionsFields.SUBS_RECEIVED_DATE.stored(),
                    SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.stored(),
                    SubscriptionsFields.SUBS_NAMES.stored())));

    private static final SubsMailActionModuleFactory MAIL_ACTION_FACTORY
        = new SubsMailActionModuleFactory(PRESERVE_FIELDS, true);
    private static final SubsStoreIndexModuleFactory STORE_ACTION_FACTORY
        = new SubsStoreIndexModuleFactory(PRESERVE_FIELDS, true);
    private static final int URL_PREFIX_SIZE = 6;

    private static final Set<ChangeType> ACCEPTED_CHANGE_TYPES =
        new HashSet<>(
            Arrays.asList(
                ChangeType.STORE,
                ChangeType.MOVE,
                ChangeType.FIELDS_UPDATE));

    private NewSubscriptionsModuleFactory() {
    }

    @Override
    public IndexModule create(
        final ChangeContext context)
        throws HttpException, JsonException
    {
        if (context.slow()
            || !context.kamaji().config().subscriptionsIndexEnabled()
            || !ACCEPTED_CHANGE_TYPES.contains(context.changeType()))
        {
            return null;
        }

        PrefixedLogger logger = context.session().logger();
        IndexModule module = null;
        try {
            switch (context.changeType()) {
                case STORE:
                    module =
                        STORE_ACTION_FACTORY.create(context);
                    break;
                case FIELDS_UPDATE:
                    module =
                        MAIL_ACTION_FACTORY.create(context);
                    break;
                default:
                    module = null;
            }
        } catch (JsonException je) {
            logger.log(
                Level.WARNING,
                "Failed to create subscriptions index module " + context.json(),
                je);
        }

        return module;
    }

    public static String url(
        final long prefix,
        final String email,
        final String date)
    {
        StringBuilder sb = new StringBuilder(
            email.length() + date.length() + URL_PREFIX_SIZE);
        sb.append("subs_");
        sb.append(prefix);
        sb.append('_');
        sb.append(email);
        return sb.toString();
    }

    public static boolean indexableSubscription(final MailMetaInfo meta) {
        for (Integer mtype: INDEXED_MESSAGE_TYPES) {
            if (meta.messageTypes().contains(mtype)) {
                return true;
            }
        }

        return false;
    }
}
