package ru.yandex.search.mail.kamaji.subscriptions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.MutableDateTime;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.dbfields.SubscriptionsIndexField;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.ps.mail.search.SubscriptionsFields;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.mail.kamaji.KamajiIndexationContext;
import ru.yandex.search.mail.kamaji.senders.AbstractSendersIndexerModule;
import ru.yandex.search.mail.kamaji.senders.Address;
import ru.yandex.search.mail.kamaji.update.IndexationContext;

public abstract class SubsBaseModule implements IndexModule {
    private static final DateTimeZone TIMEZONE =
        DateTimeZone.forID("Europe/Moscow");
    private static final String FUNCTION = "function";
    private static final String ARGS = "args";
    private static final int THOUSAND = 1000;

    protected final PrefixedLogger logger;

    private final Set<String> preservedFields;
    private final boolean newFormat;

    protected SubsBaseModule(
        final PrefixedLogger logger,
        final Set<String> preservedFields,
        final boolean newFormat)
    {
        this.logger = logger;
        this.preservedFields = preservedFields;
        this.newFormat = newFormat;
    }

    @Override
    public Collection<String> preserveFields() {
        return preservedFields;
    }

    protected List<Integer> sortedTypes(final MailMetaInfo meta) {
        List<Integer> types = new ArrayList<>(meta.messageTypes());
        Collections.sort(types);
        return types;
    }

    protected String typesToSet(final Collection<Integer> intTypes) {
        StringBuilder sb = new StringBuilder();
        for (Integer type: intTypes) {
            sb.append(type);
            sb.append('\n');
        }

        if (sb.length() > 0) {
            sb.setLength(sb.length() - 1);
        }

        return sb.toString();
    }

    protected String typesToMap(
        final Collection<Integer> intTypes,
        final long value)
    {
        StringBuilder sb = new StringBuilder();
        for (Integer type: intTypes) {
            sb.append(type);
            sb.append('\t');
            sb.append(value);
            sb.append('\n');
        }
        if (intTypes.size() > 0) {
            sb.setLength(sb.length() - 1);
        }

        return sb.toString();
    }

    public static String normalizeEmail(final String email) {
        int atIndex = email.indexOf('@');
        if (atIndex <= 0) {
            return email;
        }

        String normalized = email;
        int plusIndex = email.indexOf('+');
        if (plusIndex > 0 && plusIndex < atIndex) {
            normalized =
                email.substring(0, plusIndex) + email.substring(atIndex);
        }

        return normalized;
    }

    protected boolean goodFolderType(final KamajiIndexationContext context) {
        if (context.folderType() == null) {
            return false;
        }

        boolean status;
        switch (context.folderType()) {
            case INBOX:
            case USER:
            case PENDING:
                status = true;
                break;
            case TRASH:
            case SPAM:
                status = false;
                break;
            default:
                status = false;
        }

        return status;
    }


    // CSOFF: ReturnCount
    protected List<Map<String, Object>> indexDocuments(
        final IndexationContext context,
        final String readValues,
        final String receivedValue)
        throws NumberFormatException
    {
        MailMetaInfo meta = context.meta();
        String tsStr = meta.get(MailMetaInfo.RECEIVED_DATE);
        Long ts = null;
        if (tsStr != null) {
            ts = Long.parseLong(tsStr);
        }

        List<Address> addresses =
            AbstractSendersIndexerModule.extractSender(context.meta());
        if (addresses.size() != 1) {
            return Collections.emptyList();
        }

        String names = addresses.get(0).names();

        String email =
            meta.get(
                MailMetaInfo.HDR + MailMetaInfo.FROM + MailMetaInfo.NORMALIZED);

        if (email == null || ts == null) {
            logger.warning(
                "Subscriptions no from or bad ts in meta, skipping");
            return Collections.emptyList();
        }

        email = normalizeEmail(email.trim());

        MutableDateTime receivedTs =
            new MutableDateTime(ts * THOUSAND, TIMEZONE);
        receivedTs.setDayOfMonth(1);
        DateTime dayStart = receivedTs.toDateTime().withTimeAtStartOfDay();
        long monthTs = dayStart.getMillis() / THOUSAND;

        Map<String, Object> map = new LinkedHashMap<>();
        if (newFormat) {
            map.put(
                MailIndexFields.URL,
                NewSubscriptionsModuleFactory.url(
                    context.prefix(),
                    email,
                    String.valueOf(monthTs)));

            map.put(SubscriptionsFields.SUBS_EMAIL.stored(), email);

            if (!readValues.isEmpty()) {
                map.put(
                    SubscriptionsFields.SUBS_READ_TYPES.stored(),
                    incFunciton(
                        SubscriptionsFields.SUBS_READ_TYPES.stored(),
                        readValues));
            }

            if (!receivedValue.isEmpty()) {
                map.put(
                    SubscriptionsFields.SUBS_RECEIVED_TYPES.stored(),
                    incFunciton(
                        SubscriptionsFields.SUBS_RECEIVED_TYPES.stored(),
                        receivedValue));
            }

            if (names != null) {
                map.put(SubscriptionsFields.SUBS_NAMES.stored(), names);
            }

            map.put(
                SubscriptionsFields.SUBS_RECEIVED_DATE.stored(),
                maxFunction(
                    SubscriptionsFields.SUBS_RECEIVED_DATE.stored(),
                    tsStr));
            map.put(SubscriptionsFields.SUBS_UID.stored(), context.prefix());
            map.put(SubscriptionsFields.SUBS_DOMAIN.stored(), MailAliases.INSTANCE.parseAndNormalize(email).getValue());
        } else {
            map.put(
                MailIndexFields.URL,
                SubsIndexModuleFactory.url(
                    context.prefix(),
                    email,
                    String.valueOf(monthTs)));

            map.put(MailIndexFields.TYPE, "mail_subscriptions");
            map.put(SubscriptionsIndexField.SUBS_EMAIL.fieldName(), email);

            if (!readValues.isEmpty()) {
                map.put(
                    SubscriptionsIndexField.SUBS_READ_TYPES_MAP.fieldName(),
                    incFunciton(
                        SubscriptionsIndexField.SUBS_READ_TYPES_MAP.fieldName(),
                        readValues));
            }

            if (!receivedValue.isEmpty()) {
                map.put(
                    SubscriptionsIndexField.SUBS_RECEIVED_TYPES_MAP.fieldName(),
                    incFunciton(
                        SubscriptionsIndexField.SUBS_RECEIVED_TYPES_MAP.fieldName(),
                        receivedValue));
            }

            if (names != null) {
                map.put(SubscriptionsIndexField.SUBS_NAME.fieldName(), names);
            }

            map.put(
                SubscriptionsIndexField.SUBS_RECEIVED_MONTH.fieldName(),
                monthTs);
            map.put(
                SubscriptionsIndexField.SUBS_LAST_RECEIVED_DATE.fieldName(),
                maxFunction(
                    SubscriptionsIndexField.SUBS_LAST_RECEIVED_DATE.fieldName(),
                    tsStr));
        }



        return Collections.singletonList(map);
    }
    // CSON: ReturnCount

    protected Map<String, Object> incFunciton(
        final String field,
        final String value)
    {
        List<Object> sumMapArgs = new ArrayList<>(2);
        sumMapArgs.add(value);
        sumMapArgs.add(getFunction(field));

        Map<String, Object> incFunc = new LinkedHashMap<>();
        incFunc.put(FUNCTION, "sum_map");
        incFunc.put(ARGS, sumMapArgs);
        return incFunc;
    }

    protected Map<String, Object> getFunction(final String field) {
        Map<String, Object> getFunc = new LinkedHashMap<>();
        getFunc.put(FUNCTION, "get");
        getFunc.put(ARGS, Collections.singletonList(field));
        return getFunc;
    }

    protected Map<String, Object> maxFunction(
        final String field,
        final String value)
    {
        List<Object> maxArgs = new ArrayList<>(2);

        maxArgs.add(value);
        maxArgs.add(getFunction(field));

        Map<String, Object> incFunc = new LinkedHashMap<>();
        incFunc.put(FUNCTION, "max");
        incFunc.put(ARGS, maxArgs);
        return incFunc;
    }
}
