package ru.yandex.search.mail.kamaji.subscriptions;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.SubscriptionsIndexField;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.kamaji.ChangeContext;

public final class SubsIndexModuleFactory implements IndexModuleFactory {
    public static final SubsIndexModuleFactory INSTANCE
        = new SubsIndexModuleFactory();

    public static final Set<String> PRESERVE_FIELDS =
        Collections.unmodifiableSet(
            new HashSet<>(
                Arrays.asList(
                    SubscriptionsIndexField.SUBS_READ_TYPES_MAP.fieldName(),
                    SubscriptionsIndexField.SUBS_RECEIVED_TYPES_MAP.fieldName(),
                    SubscriptionsIndexField.SUBS_LAST_RECEIVED_DATE.fieldName(),
                    SubscriptionsIndexField.SUBS_NAME.fieldName())));

    private static final SubsMailActionModuleFactory MAIL_ACTION_FACTORY
        = new SubsMailActionModuleFactory(PRESERVE_FIELDS, false);
    private static final SubsStoreIndexModuleFactory STORE_ACTION_FACTORY
        = new SubsStoreIndexModuleFactory(PRESERVE_FIELDS, false);

    private static final int URL_PREFIX_SIZE = 6;

    private static final Set<ChangeType> ACCEPTED_CHANGE_TYPES =
        new HashSet<>(
            Arrays.asList(
                ChangeType.STORE,
                ChangeType.MOVE,
                ChangeType.FIELDS_UPDATE));

    private SubsIndexModuleFactory() {
    }

    @Override
    public IndexModule create(
        final ChangeContext context)
        throws HttpException, JsonException
    {
        if (context.slow()
            || !context.kamaji().config().subscriptionsIndexEnabled()
            || !ACCEPTED_CHANGE_TYPES.contains(context.changeType()))
        {
            return null;
        }

        PrefixedLogger logger = context.session().logger();
        IndexModule module = null;
        try {
            switch (context.changeType()) {
                case STORE:
                    module = STORE_ACTION_FACTORY.create(context);
                    break;
                case FIELDS_UPDATE:
                    module = MAIL_ACTION_FACTORY.create(context);
                    break;
                default:
                    module = null;
            }
        } catch (JsonException je) {
            logger.log(
                Level.WARNING,
                "Failed to create subscriptions index module " + context.json(),
                je);
        }

        return module;
    }

    public static String url(
        final long prefix,
        final String email,
        final String date)
    {
        StringBuilder sb = new StringBuilder(
            email.length() + date.length() + URL_PREFIX_SIZE);
        sb.append("subs_");
        sb.append(prefix);
        sb.append('_');
        sb.append(email);
        sb.append('_');
        sb.append(date);
        return sb.toString();
    }
}
