package ru.yandex.search.mail.kamaji.update;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.search.document.mail.FirstlineMailMetaInfo;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.mail.kamaji.ChangeContext;
import ru.yandex.search.mail.kamaji.OffsetCallback;
import ru.yandex.search.mail.kamaji.senders.AbstractSendersIndexerModule;
import ru.yandex.search.mail.kamaji.senders.Address;
import ru.yandex.search.mail.kamaji.usertype.UserTypeChecker;
import ru.yandex.search.mail.kamaji.usertype.UserTypeChecker.CheckType;

public abstract class AbstractFilterSearchCallback<T extends
    MailPreparedCallback>
    implements FutureCallback<Map.Entry<List<FirstlineMailMetaInfo>, Object>>
{
    protected final ChangeContext context;
    protected final UserTypeChecker userTypeChecker;
    protected final Map<String, T> callbackMap;

    private final String uri;

    private boolean done;

    protected AbstractFilterSearchCallback(
        final OffsetCallback offsetCallback,
        final String uri)
    {
        this.context = offsetCallback.context();
        this.uri = uri;

        this.callbackMap = new HashMap<>(offsetCallback.batchSize());
        this.userTypeChecker = new UserTypeChecker(this);
    }

    public ChangeContext context() {
        return context;
    }

    public void onMeta(final MailMetaInfo meta) {
        T fmpCallback;
        String mid;
        try {
            mid = ValueUtils.asString(meta.get(MailMetaInfo.MID));

            synchronized (this) {
                if (done) {
                    return;
                }

                fmpCallback = this.callbackMap.remove(mid);
            }
        } catch (JsonUnexpectedTokenException e) {
            failed(e);
            return;
        }

        if (fmpCallback != null) {
            fmpCallback.metaReady(meta);
        } else {
            context.session().logger().info(
                "Filter search returned unexpected mid:" + mid);
        }
    }

    @Override
    public void cancelled() {
        synchronized (this) {
            if (done) {
                return;
            }

            done = true;
        }

        for (T cb: callbackMap.values()) {
            cb.cancelled();
        }
    }

    @Override
    public void failed(final Exception e) {
        synchronized (this) {
            if (done) {
                return;
            }

            done = true;
        }

        for (T cb: callbackMap.values()) {
            cb.failed(e);
        }
    }

    public void completed() {
        synchronized (this) {
            if (done) {
                return;
            }

            done = true;
        }

        // cleanup locks, complete callbacks for missing mids
        for (Map.Entry<String, T> cb: callbackMap.entrySet()) {
            cb.getValue().metaGone();
        }
    }

    @Override
    public void completed(
        final Map.Entry<List<FirstlineMailMetaInfo>, Object> result)
    {
        try {
            List<FirstlineMailMetaInfo> metas = result.getKey();
            if (metas.isEmpty()) {
                context.session().logger().info(
                    "All mids from " + uri + " have gone");
                completed();
            } else {
                List<MailMetaInfo> metasToIndex = new ArrayList<>(metas.size());
                List<MailMetaInfo> threadCheck = new ArrayList<>(metas.size());
                Map<String, List<MailMetaInfo>> mailMap = new HashMap<>();

                Object userTypeObj =
                    context.json().get(MailIndexFields.USER_TYPE);

                boolean userTypeEnabled =
                    context.kamaji().config()
                        .userTypeConfig().enabled()
                        || context.session().params()
                        .getBoolean("xavier", false);

                if (!userTypeEnabled || userTypeObj != null) {
                    String userType = null;
                    if (userTypeObj != null) {
                        userType = ValueUtils.asString(userTypeObj);
                    }

                    for (MailMetaInfo meta : metas) {
                        if (userType != null) {
                            meta.set(MailIndexFields.USER_TYPE, userType);
                        }

                        meta.set(
                            MailIndexFields.QUEUE_ID,
                            String.valueOf(context.zooQueueId()));

                        metasToIndex.add(meta);
                    }
                } else {
                    for (MailMetaInfo meta : metas) {
                        meta.set(
                            MailIndexFields.QUEUE_ID,
                            String.valueOf(context.zooQueueId()));

                        List<Address> addresses =
                            AbstractSendersIndexerModule.extractSender(meta);

                        CheckType userCheckType =
                            userTypeChecker.getCheck(meta, addresses);

                        switch (userCheckType) {
                            case THREAD:
                                threadCheck.add(meta);
                                break;
                            case FROM:
                                String email = addresses.get(0).normalized();
                                List<MailMetaInfo> metaList =
                                    mailMap.get(email);

                                if (metaList == null) {
                                    metaList = new ArrayList<>(metas.size());
                                    mailMap.put(email, metaList);
                                }

                                metaList.add(meta);
                                break;
                            default:
                                metasToIndex.add(meta);
                                break;
                        }
                    }
                }

                for (MailMetaInfo meta: metasToIndex) {
                    onMeta(meta);
                }

                if (threadCheck.size() > 0 || mailMap.size() > 0) {
                    userTypeChecker.extractUserTypes(threadCheck, mailMap);
                } else {
                    completed();
                }
            }
        } catch (HttpException | JsonUnexpectedTokenException e) {
            failed(e);
        }
    }
}
